package ru.yandex.solomon.alert.inject.spring;

import java.time.Clock;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.solomon.alert.api.validators.NotificationValidator;
import ru.yandex.solomon.alert.cluster.AlertingHeartbeatProcess;
import ru.yandex.solomon.alert.cluster.AlertingShardProxy;
import ru.yandex.solomon.alert.cluster.balancer.AlertingBalancer;
import ru.yandex.solomon.alert.cluster.balancer.AlertingBalancerProxy;
import ru.yandex.solomon.alert.cluster.balancer.AlertingLocalShards;
import ru.yandex.solomon.alert.cluster.balancer.AlertingLocalShardsImpl;
import ru.yandex.solomon.alert.cluster.balancer.ShardsHolderImpl;
import ru.yandex.solomon.alert.cluster.balancer.client.AlertingBalancerClient;
import ru.yandex.solomon.alert.cluster.balancer.client.AlertingBalancerClientImpl;
import ru.yandex.solomon.alert.cluster.balancer.leader.AlertingResource;
import ru.yandex.solomon.alert.cluster.broker.AlertingProjectShard;
import ru.yandex.solomon.alert.cluster.broker.AlertingProjectShardFactory;
import ru.yandex.solomon.alert.cluster.broker.evaluation.EvaluationAssignmentService;
import ru.yandex.solomon.alert.cluster.discovery.AlertingClusterDiscovery;
import ru.yandex.solomon.alert.cluster.discovery.AlertingTransports;
import ru.yandex.solomon.alert.dao.AlertStatesDao;
import ru.yandex.solomon.alert.dao.EntitiesDao;
import ru.yandex.solomon.alert.dao.ProjectHolderImpl;
import ru.yandex.solomon.alert.dao.ProjectsHolder;
import ru.yandex.solomon.alert.dao.ShardsDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.mute.domain.Mute;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.auth.http.HttpAuthenticator;
import ru.yandex.solomon.auth.internal.InternalAuthorizer;
import ru.yandex.solomon.balancer.BalancerHolder;
import ru.yandex.solomon.balancer.BalancerHolderImpl;
import ru.yandex.solomon.balancer.ShardsHolder;
import ru.yandex.solomon.balancer.TotalShardCounter;
import ru.yandex.solomon.balancer.TotalShardCounterImpl;
import ru.yandex.solomon.balancer.dao.BalancerDao;
import ru.yandex.solomon.balancer.remote.RemoteCluster;
import ru.yandex.solomon.balancer.remote.RemoteClusterImpl;
import ru.yandex.solomon.balancer.remote.RemoteNodeClient;
import ru.yandex.solomon.balancer.remote.RemoteNodeClientImpl;
import ru.yandex.solomon.balancer.www.BalancerController;
import ru.yandex.solomon.config.protobuf.http.HttpServerConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.staffOnly.RootLink;
import ru.yandex.solomon.staffOnly.manager.ManagerWriterContext;

/**
 * @author Vladimir Gordiychuk
 */
@Import({
    AlertingDiscoveryContext.class,
    ValidatorsContext.class,
    ManagerWriterContext.class
})
@Configuration
public class AlertingBalancerContext {
    private final Clock clock;
    private final DistributedLock lock;
    private final ScheduledExecutorService timer;
    private final ExecutorService executor;

    public AlertingBalancerContext(Clock clock, DistributedLock lock, ThreadPoolProvider provider) {
        this.clock = clock;
        this.lock = lock;
        this.timer = provider.getSchedulerExecutorService();
        this.executor = provider.getExecutorService("CpuHighPriority", "");
    }

    @Bean
    public AlertingBalancerClient balancerClient(ClusterDiscovery<AlertingTransports> discovery) {
        return new AlertingBalancerClientImpl(discovery);
    }

    @Bean
    public TotalShardCounter totalShardCounter() {
        return new TotalShardCounterImpl();
    }

    @Bean
    public RemoteNodeClient remoteNodeClient(
        ClusterDiscovery<AlertingTransports> discovery,
        TotalShardCounter shardCounter)
    {
        return new RemoteNodeClientImpl(AlertingClusterDiscovery.internal(discovery), shardCounter);
    }

    @Bean
    public RemoteCluster remoteCluster(RemoteNodeClient remoteNodeClient) {
        return new RemoteClusterImpl(remoteNodeClient, timer, clock);
    }

    @Bean
    public AlertingHeartbeatProcess heartbeatProcess(
        AlertingLocalShards shards,
        EvaluationAssignmentService evaluationAssignmentService,
        AlertingBalancerClient client)
    {
        return new AlertingHeartbeatProcess(clock, shards, evaluationAssignmentService, lock, client, timer, executor);
    }

    @Bean
    public ProjectsHolder projectsHolder(ProjectsDao projectsDao) {
        return new ProjectHolderImpl(projectsDao, executor, timer);
    }

    @Bean
    public ShardsHolder shardsHolder(
            ProjectsHolder projectsHolder,
            ShardsDao shards,
            EntitiesDao<Alert> alerts,
            EntitiesDao<Notification> notifications,
            EntitiesDao<Mute> mutes,
            AlertStatesDao states)
    {
        return new ShardsHolderImpl(projectsHolder, shards, alerts, notifications, mutes, states, lock);
    }

    @Bean
    public BalancerHolderImpl balancerHolder(
        RemoteCluster cluster,
        ShardsHolder shardsHolder,
        TotalShardCounter shardCounter,
        BalancerDao dao)
    {
        return new BalancerHolderImpl(
            clock,
            lock,
            AlertingResource.RESOURCES,
            cluster,
            executor,
            timer,
            shardsHolder,
            shardCounter,
            dao);
    }

    @Bean
    public AlertingBalancerProxy alertingBalancerProxy(BalancerHolder balancerHolder, RemoteCluster cluster, AlertingBalancerClient client) {
        return new AlertingBalancerProxy(clock, balancerHolder, cluster, client);
    }

    @Bean
    public AlertingLocalShardsImpl localShardsState(
        AlertingProjectShardFactory shardFactory,
        AlertingBalancerClient client)
    {
        return new AlertingLocalShardsImpl(shardFactory, client, lock, timer, executor);
    }

    @Bean
    public AlertingShardProxy alertingShardProxy(
            AlertingLocalShards localShards,
            AlertingBalancer balancer,
            ClusterDiscovery<AlertingTransports> discovery,
            NotificationValidator notificationValidator)
    {
        return new AlertingShardProxy(localShards, balancer, discovery, notificationValidator);
    }

    @Bean
    public Bootstrap balancerBootstrap(BalancerHolderImpl balancer, AlertingHeartbeatProcess heartbeatProcess) {
        return new Bootstrap(balancer, heartbeatProcess, timer);
    }

    @Bean
    public BalancerController balancerController(
            HttpAuthenticator authenticator,
            InternalAuthorizer authorizer,
            BalancerHolder balancer,
            HttpServerConfig config)
    {
        return new BalancerController(authenticator, authorizer, balancer, AlertingProjectShard.class, config.getPort());
    }

    @Bean
    public RootLink balancerLink() {
        return new RootLink("/balancer", "Balancer");
    }

    public static class Bootstrap {
        private final BalancerHolderImpl balancer;
        private final AlertingHeartbeatProcess heartbeatProcess;
        private final ScheduledExecutorService timer;

        public Bootstrap(BalancerHolderImpl balancer, AlertingHeartbeatProcess heartbeatProcess, ScheduledExecutorService timer) {
            this.balancer = balancer;
            this.heartbeatProcess = heartbeatProcess;
            this.timer = timer;
        }

        @PostConstruct
        public void startupHandler() {
            balancer.start();
            timer.schedule(heartbeatProcess::start, 5, TimeUnit.SECONDS);
        }

        @PreDestroy
        public void shutdownHandler() {
            heartbeatProcess.stop();
            balancer.close();
        }
    }
}
