package ru.yandex.solomon.alert.inject.spring;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import com.google.common.net.HostAndPort;
import org.apache.logging.log4j.util.Strings;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.cluster.discovery.ClusterDiscoveryImpl;
import ru.yandex.discovery.DiscoveryService;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.conf.GrpcConfigurationContext;
import ru.yandex.grpc.utils.DefaultClientOptions;
import ru.yandex.grpc.utils.GrpcClientOptions;
import ru.yandex.grpc.utils.GrpcTransport;
import ru.yandex.solomon.alert.cluster.discovery.AlertingTransports;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.protobuf.Time;
import ru.yandex.solomon.config.protobuf.alert.TAlertingConfig;
import ru.yandex.solomon.config.protobuf.alert.TAlertingDiscoveryConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.ctx.ServiceAuthContext;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@Import({
        GrpcConfigurationContext.class,
        ServiceAuthContext.class
})
public class AlertingDiscoveryContext {
    private final String clientId;
    private final int internalPort;
    private final int externalPort;
    private final TAlertingDiscoveryConfig config;
    private final ThreadPoolProvider poolProvider;

    public AlertingDiscoveryContext(TAlertingConfig config, ThreadPoolProvider poolProvider) {
        this.clientId = Strings.trimToNull(config.getClientId());
        this.externalPort = config.getRpcServerConfig().getGrpcServerConfig().getPort(0);
        this.internalPort = config.getRpcServerConfig().getInternalGrpcServerConfig().getPort(0);
        this.config = config.getAlertingDiscoveryConfig();
        this.poolProvider = poolProvider;
    }

    @Bean
    public ClusterDiscovery<AlertingTransports> alertingDiscovery(
            ClientOptionsFactory clientOptionsFactory)
    {
        var addresses = config.getGrpcClientConfig().getAddressesList();
        var ops = makeTransportOpts(clientOptionsFactory);
        ScheduledExecutorService timer = poolProvider.getSchedulerExecutorService();
        ExecutorService executor = poolProvider.getExecutorService(config.getThreadPoolName(), "AlertingDiscoveryConfig.ThreadPoolName");
        long reloadMillis = getReloadIntervalMillis();
        return new ClusterDiscoveryImpl<>(hostAndPort -> makeTransport(ops, hostAndPort), addresses, DiscoveryService.async(), timer, executor, reloadMillis);
    }

    private AlertingTransports makeTransport(GrpcClientOptions opts, HostAndPort hostAndPort) {
        if (hostAndPort.hasPort()) {
            var transport = new GrpcTransport(hostAndPort, opts);
            return new AlertingTransports(transport, transport);
        }

        return new AlertingTransports(
                new GrpcTransport(hostAndPort.withDefaultPort(externalPort), opts),
                new GrpcTransport(hostAndPort.withDefaultPort(internalPort), opts));
    }

    private DefaultClientOptions makeTransportOpts(ClientOptionsFactory factory) {
        return factory.newBuilder("AlertingDiscoveryConfig.GrpcClientConfig", config.getGrpcClientConfig())
                .setClientId(clientId)
                .build();
    }

    private long getReloadIntervalMillis() {
        Time time = config.getRefreshInterval();
        if (time.getUnitValue() == 0) {
            return TimeUnit.HOURS.toMillis(1);
        }
        return TimeUnitConverter.protoToUnit(time.getUnit()).toMillis(time.getValue());
    }
}
