package ru.yandex.solomon.alert.inject.spring;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillClose;

import io.grpc.BindableService;
import io.grpc.Server;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.grpc.utils.GrpcServerFactory;
import ru.yandex.solomon.alert.cluster.server.grpc.GrpcAlertingClusterService;
import ru.yandex.solomon.alert.cluster.server.grpc.GrpcBalancerService;
import ru.yandex.solomon.alert.grpc.GrpcAlertService;
import ru.yandex.solomon.alert.grpc.GrpcAlertTemplateService;
import ru.yandex.solomon.alert.grpc.GrpcMuteService;
import ru.yandex.solomon.alert.grpc.GrpcNotificationService;
import ru.yandex.solomon.alert.grpc.GrpcTelegramService;
import ru.yandex.solomon.alert.grpc.GrpcYaChatsService;
import ru.yandex.solomon.config.protobuf.rpc.TGrpcServerConfig;
import ru.yandex.solomon.config.protobuf.rpc.TRpcServerConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.grpc.GrpcContext;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@Import({
    GrpcContext.class,
    AlertingGrpcServiceContext.class
})
@ParametersAreNonnullByDefault
public class AlertingGrpcServerContext implements DisposableBean {
    private final TRpcServerConfig config;
    private final GrpcServerFactory serverFactory;
    private final ThreadPoolProvider threads;

    @WillClose
    private Server externalServer;
    @WillClose
    private Server internalServer;

    public AlertingGrpcServerContext(TRpcServerConfig config, GrpcServerFactory serverFactory, ThreadPoolProvider threads) {
        this.config = config;
        this.serverFactory = serverFactory;
        this.threads = threads;
    }

    @Bean
    public Server grpcExternalServer(
        GrpcAlertService alerting,
        GrpcNotificationService notification,
        GrpcTelegramService telegram,
        GrpcMuteService mute,
        GrpcAlertTemplateService templateService,
        Optional<GrpcYaChatsService> yaChatsO)
    {
        String prefix = "RpcServerConfig.GrpcServerConfig";
        TGrpcServerConfig config = this.config.getGrpcServerConfig();

        List<BindableService> serviceList = new ArrayList<>();
        serviceList.add(alerting);
        serviceList.add(notification);
        serviceList.add(mute);
        serviceList.add(telegram);
        serviceList.add(templateService);
        yaChatsO.ifPresent(serviceList::add);

        externalServer = serverFactory.makeServer(prefix, config, serviceList);
        serverFactory.startWithTimeout(config, externalServer);
        return externalServer;
    }

    @Bean
    public Server grpcInternalServer(
            GrpcAlertingClusterService service,
            GrpcBalancerService balancer,
            GrpcBalancerService.Proxy balancerProxy)
    {
        String prefix = "RpcServerConfig.InternalGrpcServerConfig";
        TGrpcServerConfig config = this.config.getInternalGrpcServerConfig();
        internalServer = serverFactory.makeServer(prefix, config, service, balancer, balancerProxy);
        serverFactory.startWithTimeout(config, internalServer);
        return internalServer;
    }

    @Override
    public void destroy() {
        if (externalServer != null) {
            externalServer.shutdown();
        }

        if (internalServer != null) {
            internalServer.shutdown();
        }
    }
}
