package ru.yandex.solomon.alert.inject.spring;

import java.nio.file.Path;
import java.time.Clock;
import java.time.Duration;
import java.util.Collection;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.springframework.beans.factory.annotation.Configurable;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;

import ru.yandex.cluster.discovery.ClusterDiscovery;
import ru.yandex.jns.client.JnsClient;
import ru.yandex.jns.client.JnsClientStub;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.api.converters.MuteConverter;
import ru.yandex.solomon.alert.api.converters.NotificationConverter;
import ru.yandex.solomon.alert.cluster.broker.AlertingProjectShardFactory;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.TemplateAlertFactory;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.search.ActivityFilters;
import ru.yandex.solomon.alert.cluster.broker.alert.activity.search.ActivitySearch;
import ru.yandex.solomon.alert.cluster.broker.evaluation.EvaluationAssignmentService;
import ru.yandex.solomon.alert.cluster.broker.evaluation.EvaluationAssignmentServiceImpl;
import ru.yandex.solomon.alert.cluster.broker.mute.search.MuteSearch;
import ru.yandex.solomon.alert.cluster.broker.notification.search.NotificationSearch;
import ru.yandex.solomon.alert.cluster.discovery.AlertingClusterDiscovery;
import ru.yandex.solomon.alert.cluster.discovery.AlertingTransports;
import ru.yandex.solomon.alert.dao.AlertStatesDao;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.AlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.dao.DaoFactory;
import ru.yandex.solomon.alert.dao.DaoFactoryImpl;
import ru.yandex.solomon.alert.dao.EntitiesDao;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.inject.spring.convert.RetryOptionsConverter;
import ru.yandex.solomon.alert.inject.spring.notification.ChartsContext;
import ru.yandex.solomon.alert.inject.spring.notification.EmailContext;
import ru.yandex.solomon.alert.inject.spring.notification.FallbackContext;
import ru.yandex.solomon.alert.inject.spring.notification.GeneralContext;
import ru.yandex.solomon.alert.inject.spring.notification.JugglerContext;
import ru.yandex.solomon.alert.inject.spring.notification.PhoneContext;
import ru.yandex.solomon.alert.inject.spring.notification.SmsContext;
import ru.yandex.solomon.alert.inject.spring.notification.TelegramContext;
import ru.yandex.solomon.alert.inject.spring.notification.WebhookContext;
import ru.yandex.solomon.alert.inject.spring.notification.YaChatsContext;
import ru.yandex.solomon.alert.mute.domain.Mute;
import ru.yandex.solomon.alert.notification.RetryOptions;
import ru.yandex.solomon.alert.notification.channel.CompositeNotificationChannelFactory;
import ru.yandex.solomon.alert.notification.channel.NotificationChannelFactory;
import ru.yandex.solomon.alert.notification.channel.telegram.ChatIdStorage;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.state.StatefulNotificationChannelFactory;
import ru.yandex.solomon.alert.notification.state.StatefulNotificationChannelFactoryImpl;
import ru.yandex.solomon.alert.unroll.MultiAlertUnrollFactory;
import ru.yandex.solomon.alert.unroll.MultiAlertUnrollFactoryImpl;
import ru.yandex.solomon.alert.unroll.UnrollExecutor;
import ru.yandex.solomon.alert.unroll.UnrollExecutorImpl;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.alert.BrokerConfig;
import ru.yandex.solomon.config.protobuf.alert.UnrollConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.db.dao.QuotasDao;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.idempotency.IdempotentOperationService;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.quotas.watcher.QuotaWatcher;
import ru.yandex.solomon.util.file.FileStorage;
import ru.yandex.solomon.util.file.SimpleFileStorage;

import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;
import static ru.yandex.solomon.config.TimeConverter.protoToDuration;

/**
 * @author Vladimir Gordiychuk
 */
@Configurable
@Import({
    GeneralContext.class,
    EmailContext.class,
    JugglerContext.class,
    WebhookContext.class,
    PhoneContext.class,
    SmsContext.class,
    TelegramContext.class,
    FallbackContext.class,
    ChartsContext.class,
    YaChatsContext.class,
    DatalensEmailContext.class,
    CloudChannelsContext.class
})
public class BrokerContext {
    private final BrokerConfig config;
    private final ScheduledExecutorService timer;
    private final ExecutorService executor;
    private final ForkJoinPool selfMonFjp;
    private final Clock clock;
    private final FileStorage storage;
    private static final Path CACHE_DIR = Path.of("cache");

    public BrokerContext(BrokerConfig config, Clock clock, ThreadPoolProvider threadPool) {
        this.config = config;
        this.clock = clock;
        this.timer = threadPool.getSchedulerExecutorService();
        this.executor = threadPool.getExecutorService("CpuLowPriority", "");

        var metricsFjpName = config.getSelfMonForkJoinPoolName().isEmpty()
            ? "CpuLowPriority"
            : config.getSelfMonForkJoinPoolName();
        this.selfMonFjp = (ForkJoinPool) threadPool.getExecutorService(
            metricsFjpName,
            "BrokerConfig.SelfMonForkJoinPoolName");

        this.storage = new SimpleFileStorage(CACHE_DIR);
    }

    @Bean
    public DaoFactory daoFactory(
            EntitiesDao<Alert> alertsDao,
            EntitiesDao<Notification> notificationsDao,
            EntitiesDao<Mute> mutesDao,
            AlertStatesDao statesDao)
    {
        return new DaoFactoryImpl(alertsDao, notificationsDao, mutesDao, statesDao);
    }

    @Bean
    public NotificationConverter notificationConverter(ChatIdStorage chatIdResolver) {
        return new NotificationConverter(chatIdResolver);
    }

    @Bean
    public NotificationSearch notificationSearch(NotificationConverter notificationConverter) {
        return new NotificationSearch(notificationConverter);
    }

    @Bean
    public ActivityFilters activityFilters(NotificationConverter notificationConverter) {
        return new ActivityFilters(notificationConverter);
    }

    @Bean
    public ActivitySearch activitySearch(ActivityFilters activityFilters) {
        return new ActivitySearch(activityFilters);
    }

    @Bean
    public MuteConverter muteConverter() {
        return MuteConverter.INSTANCE;
    }

    @Bean
    public MuteSearch muteSearch(MuteConverter muteConverter) {
        return new MuteSearch(muteConverter);
    }

    @Bean
    private StatefulNotificationChannelFactory statefulNotificationChannelFactory(
        NotificationConverter notificationConverter)
    {
        RetryOptions retryConfig = RetryOptionsConverter.protoToOptions(
                config.getNotificationConfig()
                        .getRetryConfig());

        return new StatefulNotificationChannelFactoryImpl(timer, retryConfig, notificationConverter);
    }

    @Bean
    public CompositeNotificationChannelFactory compositeNotificationChannelFactory(Collection<NotificationChannelFactory> factories)
    {
        return factories.stream()
            .filter(Objects::nonNull)
            .collect(collectingAndThen(toList(), CompositeNotificationChannelFactory::new));
    }

    @Bean
    QuotaWatcher quotaWatcher(QuotasDao quotasDao) {
        return new QuotaWatcher(storage, quotasDao, "alerting", timer);
    }

    @Bean
    public AlertingProjectShardFactory shardFactory(
            DaoFactory dao,
            CompositeNotificationChannelFactory factories,
            StatefulNotificationChannelFactory statefulChannelFactory,
            UnrollExecutor unrollExecutor,
            EvaluationAssignmentService assignmentService,
            NotificationConverter notificationConverter,
            NotificationSearch notificationSearch,
            ActivitySearch activitySearch,
            MuteConverter muteConverter,
            MuteSearch muteSearch,
            QuotaWatcher quotaWatcher,
            AlertTemplateDao cachedAlertTemplateDao,
            AlertTemplateLastVersionDao alertTemplateLastVersionDao,
            TemplateAlertFactory templateFactory,
            MetricRegistry registry,
            IdempotentOperationService idempotentOperationService,
            FeatureFlagsHolder flagsHolder,
            Optional<JnsClient> clientOptional)
    {
        Duration snapshotInterval = TimeConverter.protoToDuration(config.getSnapshotInterval(), Duration.ofSeconds(150));

        return new AlertingProjectShardFactory(
            clock,
            dao,
            factories,
            statefulChannelFactory,
            unrollExecutor,
            assignmentService,
            executor,
            timer,
            snapshotInterval,
            notificationConverter,
            notificationSearch,
            activitySearch,
            muteConverter,
            muteSearch,
            quotaWatcher,
            cachedAlertTemplateDao,
            alertTemplateLastVersionDao,
            templateFactory,
            registry,
            idempotentOperationService,
            flagsHolder,
            selfMonFjp,
            clientOptional.orElse(new JnsClientStub()));
    }

    @Bean
    public EvaluationAssignmentService evaluationAssignmentService(
        ClusterDiscovery<AlertingTransports> discovery,
        MetricRegistry registry)
    {
        return new EvaluationAssignmentServiceImpl(
            AlertingClusterDiscovery.internal(discovery),
            timer,
            executor,
            registry,
            TimeUnit.MINUTES.toMillis(2));
    }

    @Bean
    public UnrollExecutorImpl unrollExecutor(MultiAlertUnrollFactory factory, MetricRegistry registry) {
        UnrollConfig config = this.config.getUnrollConfig();
        return new UnrollExecutorImpl(
                clock,
                timer,
                registry,
                factory,
                protoToDuration(config.getUnrollInterval(), Duration.ofMinutes(8)),
                protoToDuration(config.getUnrollTimeout(), Duration.ofSeconds(30)));
    }

    @Bean
    public MultiAlertUnrollFactory unrollFactory(MetricsClient client) {
        return new MultiAlertUnrollFactoryImpl(client);
    }

}
