package ru.yandex.solomon.alert.inject.spring;

import java.time.Clock;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.cluster.server.grpc.AssignmentTracker;
import ru.yandex.solomon.alert.cluster.server.grpc.EvaluationServerStatusHandler;
import ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationService;
import ru.yandex.solomon.alert.evaluation.AlertExecutorOptionsProvider;
import ru.yandex.solomon.alert.evaluation.AlertRuleExecutorType;
import ru.yandex.solomon.alert.evaluation.EvaluationService;
import ru.yandex.solomon.alert.evaluation.EvaluationServiceImpl;
import ru.yandex.solomon.alert.evaluation.TaskExecutorDispatcher;
import ru.yandex.solomon.alert.executor.local.LocalAlertExecutorOptions;
import ru.yandex.solomon.alert.rule.AlertRuleFactory;
import ru.yandex.solomon.alert.rule.AlertRuleFactoryImpl;
import ru.yandex.solomon.alert.rule.usage.ProjectAlertRuleMetrics;
import ru.yandex.solomon.alert.statuses.AlertingStatusesSelectorImpl;
import ru.yandex.solomon.alert.template.TemplateFactory;
import ru.yandex.solomon.config.protobuf.alert.ExecutorConfig;
import ru.yandex.solomon.config.protobuf.alert.TAlertingStatusesShardConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.metrics.client.MetricsClient;

import static ru.yandex.solomon.config.OptionalSet.setInt;
import static ru.yandex.solomon.config.OptionalSet.setTime;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
public class ExecutorContext {
    private final ThreadPoolProvider threads;
    private final AlertExecutorOptionsProvider options;
    private final TAlertingStatusesShardConfig alertingStatusConfig;

    @Autowired
    public ExecutorContext(@Qualifier("ExecutorConfig") List<ExecutorConfig> configs, ThreadPoolProvider threads) {
        configs.stream()
                .collect(Collectors.groupingBy(ExecutorConfig::getType, Collectors.counting()))
                .forEach((key, value) -> {
                    if (value > 1) {
                        throw new IllegalArgumentException("Several executor configs specified for type " + key);
                    }
                });

        var configMap = configs.stream()
                .collect(Collectors.toMap(
                        config -> AlertRuleExecutorType.fromProto(config.getType()),
                        Function.identity()
                ));

        var defaultConfig = configMap.get(AlertRuleExecutorType.DEFAULT);

        if (defaultConfig == null) {
            throw new IllegalArgumentException("Default alert rule executor is not configured");
        }

        this.alertingStatusConfig = defaultConfig.getAlertingStatusesShardConfig();

        LocalAlertExecutorOptions defaultOptions = makeOverriddenOptions(defaultConfig, LocalAlertExecutorOptions.empty());

        // Executors other from DEFAULT override their options over default options
        var optionsMap = configMap.entrySet().stream().collect(Collectors.toMap(
                Map.Entry::getKey,
                e -> e.getKey() == AlertRuleExecutorType.DEFAULT ? defaultOptions : makeOverriddenOptions(e.getValue(), defaultOptions)
        ));

        this.threads = threads;
        this.options = makeOptionsProvider(optionsMap);
    }

    private static AlertExecutorOptionsProvider makeOptionsProvider(Map<AlertRuleExecutorType, LocalAlertExecutorOptions> optionsMap) {
        return type -> {
            var options = optionsMap.get(type);
            if (options != null) {
                return options;
            }
            return optionsMap.get(AlertRuleExecutorType.DEFAULT);
        };
    }

    private static LocalAlertExecutorOptions makeOverriddenOptions(ExecutorConfig config, LocalAlertExecutorOptions base) {
        LocalAlertExecutorOptions.Builder opts = base.toBuilder();
        setTime(opts::setAlertRuleTimeout, config.getAlertTimeout());
        setTime(opts::setEvalInterval, config.getEvalInterval());
        setInt(opts::setMaxConcurrentAlertRules, config.getMaxConcurrentAlert());
        setInt(opts::setMaxConcurrentWarmupAlertRules, config.getMaxConcurrentWarmupAlert());
        setInt(opts::setMaxAlertRules, config.getMaxAlert());
        setTime(opts::setMaxEvaluationLag, config.getMaxEvaluationLag());
        setTime(opts::setMaxEvaluationJitter, config.getMaxEvaluationJitter());
        return opts.build();
    }

    @Bean
    public AlertRuleFactory alertRuleFactory(
            MetricsClient cachingMetricsClient,
            ProjectAlertRuleMetrics metrics,
            TemplateFactory templateFactory,
            FeatureFlagsHolder featureFlags)
    {
        return new AlertRuleFactoryImpl(
                cachingMetricsClient,
                metrics,
                templateFactory,
                new AlertingStatusesSelectorImpl(getAlertingStatusesShard()),
                featureFlags);
    }

    @Bean
    public ProjectAlertRuleMetrics projectAlertRuleMetrics() {
        return new ProjectAlertRuleMetrics();
    }

    @Bean
    public AssignmentTracker assignmentTracker() {
        return new AssignmentTracker();
    }

    @Bean
    public GrpcEvaluationService grpcEvaluationService(
            EvaluationService evaluationService,
            AssignmentTracker assignmentTracker,
            EvaluationServerStatusHandler evaluationServerStatusHandler)
    {
        return new GrpcEvaluationService(evaluationService, assignmentTracker, evaluationServerStatusHandler, defaultExecutor());
    }

    @Bean
    public EvaluationServerStatusHandler evaluationServerStatusHandler(EvaluationService evaluationService) {
        var timer = threads.getSchedulerExecutorService();
        var executor = defaultExecutor();
        return new EvaluationServerStatusHandler(evaluationService, timer, executor);
    }

    @Bean
    public TaskExecutorDispatcher taskExecutorDispatcher(Clock clock, MetricRegistry registry) {
        var timer = threads.getSchedulerExecutorService();
        var executor = defaultExecutor();
        return new TaskExecutorDispatcher(clock, executor, timer, options, registry);
    }

    @Bean
    public EvaluationService evaluationService(Clock clock, AlertRuleFactory ruleFactory, TaskExecutorDispatcher executorProvider) {
        return new EvaluationServiceImpl(clock, ruleFactory, executorProvider);
    }

    @Nullable
    private ShardKey getAlertingStatusesShard() {
        String project = alertingStatusConfig.getProject();
        String cluster = alertingStatusConfig.getCluster();
        String service = alertingStatusConfig.getService();
        if (project.isEmpty() || cluster.isEmpty() || service.isEmpty()) {
            return null;
        }

        return new ShardKey(project, cluster, service);
    }

    private ExecutorService defaultExecutor() {
        return threads.getExecutorService("CpuLowPriority", "");
    }
}
