package ru.yandex.solomon.alert.inject.spring;

import java.time.Clock;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.client.AlertApi;
import ru.yandex.solomon.alert.client.NotificationApi;
import ru.yandex.solomon.alert.cluster.broker.notification.ChannelValidationService;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.AlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.dao.ProjectsHolder;
import ru.yandex.solomon.alert.stats.ChannelsMonitoringStatsExporter;
import ru.yandex.solomon.alert.stats.ResourceMonitoringStatsExporter;
import ru.yandex.solomon.config.protobuf.alert.ResourceMonitoringStatsConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.locks.LockService;
import ru.yandex.solomon.name.resolver.client.NameResolverClient;
import ru.yandex.solomon.secrets.SecretProvider;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
@Configuration
public class MonitoringStatsContext {

    @Bean
    public ResourceMonitoringStatsExporter resourceMonitoringStatsExporter(
            Optional<ResourceMonitoringStatsConfig> configOpt,
            LockService lockService,
            ThreadPoolProvider threads,
            MetricRegistry registry,
            SecretProvider secretProvider,
            AlertApi alertApi,
            ProjectsHolder projectsHolder,
            AlertTemplateDao alertTemplateDao,
            AlertTemplateLastVersionDao alertTemplateLastVersionDao,
            Optional<NameResolverClient> nameResolverClientOptional,
            ChannelValidationService channelValidationService)
    {
        if (configOpt.isEmpty() || nameResolverClientOptional.isEmpty()) {
            return null;
        }
        Optional<String> token = secretProvider.getSecret(configOpt.get().getOAuthToken());
        if (token.isEmpty()) {
            return null;
        }
        var lock = lockService.distributedLock("ResourceMonitoringStatsExporterMaster");
        var timer = threads.getSchedulerExecutorService();
        return new ResourceMonitoringStatsExporter(
                projectsHolder,
                lock,
                threads.getIOExecutor(),
                timer,
                Clock.systemUTC(),
                registry,
                configOpt.get(),
                secretProvider,
                alertApi,
                alertTemplateDao,
                alertTemplateLastVersionDao,
                nameResolverClientOptional.get(),
                channelValidationService);
    }

    @Bean
    public ChannelsMonitoringStatsExporter channelsMonitoringStatsExporter(
            Optional<ResourceMonitoringStatsConfig> configOpt,
            LockService lockService,
            ThreadPoolProvider threads,
            MetricRegistry registry,
            SecretProvider secretProvider,
            NotificationApi notificationApi,
            ProjectsHolder projectsHolder,
            ChannelValidationService channelValidationService)
    {
        if (configOpt.isEmpty()) {
            return null;
        }
        Optional<String> token = secretProvider.getSecret(configOpt.get().getOAuthToken());
        if (token.isEmpty()) {
            return null;
        }
        var lock = lockService.distributedLock("ChannelsMonitoringStatsExporterMaster");
        var timer = threads.getSchedulerExecutorService();
        return new ChannelsMonitoringStatsExporter(
                projectsHolder,
                lock,
                threads.getIOExecutor(),
                timer,
                Clock.systemUTC(),
                registry,
                configOpt.get(),
                secretProvider,
                notificationApi,
                channelValidationService);
    }

    @Bean
    public ChannelValidationService channelValidationService() {
        return new ChannelValidationService();
    }
}
