package ru.yandex.solomon.alert.inject.spring;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.discovery.DiscoveryService;
import ru.yandex.discovery.FilterDiscoveryService;
import ru.yandex.discovery.cluster.ClusterMapper;
import ru.yandex.grpc.conf.ClientOptionsFactory;
import ru.yandex.grpc.conf.GrpcConfigurationContext;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.config.protobuf.TNameResolverClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.name.resolver.client.CrossDcNameResolverClient;
import ru.yandex.solomon.name.resolver.client.NameResolverClient;
import ru.yandex.solomon.name.resolver.client.NoopNameResolverClient;
import ru.yandex.solomon.name.resolver.client.grpc.GrpcNameResolverClient;

/**
 * @author Alexey Trushkin
 */
@Configuration
@ParametersAreNonnullByDefault
@Import({
        GrpcConfigurationContext.class
})
public class NameResolverContext {

    @Bean
    public NameResolverClient nameResolverClient(
            Optional<TNameResolverClientConfig> configOpt,
            @Qualifier("ClientId") String clientId,
            MetricRegistry registry,
            ClusterMapper clusterMapper,
            ThreadPoolProvider threads,
            ClientOptionsFactory clientOptionsFactory)
    {
        if (configOpt.isEmpty()) {
            return new NoopNameResolverClient();
        }

        var clientByDest = createClients(clientId, configOpt.get(), clusterMapper, threads, registry, clientOptionsFactory);
        return (clientByDest.size() == 1)
                ? clientByDest.values().iterator().next()
                : new CrossDcNameResolverClient(clientByDest);
    }

    private Map<String, NameResolverClient> createClients(
            String clientId,
            TNameResolverClientConfig config,
            ClusterMapper clusterMapper,
            ThreadPoolProvider threads,
            MetricRegistry registry,
            ClientOptionsFactory factory)
    {
        Map<String, NameResolverClient> result = new HashMap<>();
        for (String clusterId : clusterMapper.knownClusterIds()) {
            var discovery = new FilterDiscoveryService(DiscoveryService.async(), (address) -> {
                return Objects.equals(clusterId, clusterMapper.byFqdnOrNull(address.getHost()));
            });
            var opts = factory.newBuilder(
                            "NameResolverClient",
                            config.getGrpcConfig())
                    .setClientId(clientId)
                    .setMetricRegistry(registry.subRegistry("target", clusterId))
                    .build();
            result.put(clusterId, new GrpcNameResolverClient(discovery, config.getGrpcConfig().getAddressesList(), opts));
        }
        return result;
    }
}
