package ru.yandex.solomon.alert.inject.spring;

import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;

import org.asynchttpclient.AsyncHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.cloud.token.Jwt;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.graph.GraphLoader;
import ru.yandex.solomon.alert.graph.GraphLoaderImpl;
import ru.yandex.solomon.alert.notification.channel.cloud.NotifyClientImpl;
import ru.yandex.solomon.alert.notification.channel.cloud.sms.UrlShortener;
import ru.yandex.solomon.alert.notification.channel.cloud.sms.YaCCUrlShortener;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.IamKeyAuthConfig;
import ru.yandex.solomon.config.protobuf.alert.BrokerConfig;
import ru.yandex.solomon.config.protobuf.alert.TNotifyConfig;
import ru.yandex.solomon.config.protobuf.alert.TUrlShortenerConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.metrics.client.MetricsClient;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.spring.ConditionalOnBean;
import ru.yandex.solomon.util.SolomonEnv;

/**
 * @author Ivan Tsybulin
 */
@Import(
        IamTokenClientContext.class
)
@ConditionalOnBean(TNotifyConfig.class)
public class NotifyContext {
    private static final Logger logger = LoggerFactory.getLogger(NotifyContext.class);

    private final MetricRegistry registry;
    private final ThreadPoolProvider threads;

    public NotifyContext(MetricRegistry registry, ThreadPoolProvider threads) {
        this.registry = registry;
        this.threads = threads;
    }

    @Bean
    public TokenProvider tokenProvider(Optional<SecretProvider> secretProvider, Optional<IamTokenClient> iamTokenClient, TNotifyConfig config) {
        return switch (config.getAuthMethodCase()) {
            case OAUTH_TOKEN -> provideTokenBySecret(secretProvider, config);
            case IAM_KEY_AUTH -> provideTokenByIam(iamTokenClient, config);
            default -> throw new IllegalArgumentException("Unknown auth method for notify client: " + config.getAuthMethodCase());
        };
    }

    private TokenProvider provideTokenBySecret(Optional<SecretProvider> maybeSecretProvider, TNotifyConfig config) {
        SecretProvider secretProvider = maybeSecretProvider.orElseThrow(() ->
                new IllegalStateException("Notify is configured to use oauth token but secret provider is not configured"));

        Optional<String> token = secretProvider.getSecret(config.getOauthToken());

        if (SolomonEnv.DEVELOPMENT.isActive() && token.isEmpty()) {
            logger.warn("Using fake OAuth token provider");
            return TokenProvider.of("fake-token");
        }

        return TokenProvider.of(token.orElseThrow(() ->
                new IllegalStateException("No secret with OAuth token for Notify was found")));
    }

    private TokenProvider provideTokenByIam(Optional<IamTokenClient> maybeIamTokenClient, TNotifyConfig config) {
        IamTokenClient iamTokenClient = maybeIamTokenClient.orElseThrow(() ->
                new IllegalStateException("Notify is configured to use IAM token but IamTokenClient is not configured"));

        IamKeyAuthConfig iamConfig = config.getIamKeyAuth();
        Path keyFilePath = Path.of(iamConfig.getKeyPath());
        if (SolomonEnv.DEVELOPMENT.isActive() && !Files.exists(keyFilePath)) {
            // Using fake IAM token provider is allowed only in development environment
            logger.warn("Using fake IAM token provider");
            return () -> "fake-token";
        }

        var jwtBuilder = Jwt.newBuilder()
                .withAccountId(iamConfig.getAccountId())
                .withKeyId(iamConfig.getKeyId())
                .withPrivateKey(keyFilePath)
                .withTtl(Duration.ofHours(1));

        return TokenProvider.iam(iamTokenClient, jwtBuilder, threads.getSchedulerExecutorService());
    }

    @Bean
    public NotifyClientImpl cloudNotifyClient(TokenProvider tokenProvider, AsyncHttpClient httpClient, TNotifyConfig config) {
        return new NotifyClientImpl(config.getHost(), httpClient, tokenProvider, registry, threads.getSchedulerExecutorService());
    }

    @Bean
    public GraphLoader graphLoader(MetricsClient metricsClient) {
        return new GraphLoaderImpl(metricsClient);
    }

    @Bean
    public UrlShortener urlShortener(
            @Qualifier("httpClient") AsyncHttpClient httpClient,
            BrokerConfig brokerConfig)
    {
        if (!brokerConfig.getNotificationConfig().hasUrlShortenerConfig()) {
            // fake shortener which returns the same url
            return CompletableFuture::completedFuture;
        }

        TUrlShortenerConfig config = brokerConfig.getNotificationConfig().getUrlShortenerConfig();
        Duration requestTimeout = TimeConverter.protoToDuration(config.getRequestTimeout(), Duration.ofSeconds(10));
        return new YaCCUrlShortener(config.getHost(), requestTimeout, httpClient, registry, threads.getSchedulerExecutorService());
    }
}
