package ru.yandex.solomon.alert.inject.spring;

import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.solomon.alert.cluster.balancer.AlertingBalancer;
import ru.yandex.solomon.alert.cluster.balancer.AlertingLocalShards;
import ru.yandex.solomon.alert.cluster.broker.evaluation.EvaluationAssignmentService;
import ru.yandex.solomon.alert.cluster.server.grpc.GrpcEvaluationService;
import ru.yandex.solomon.alert.unroll.UnrollExecutor;
import ru.yandex.solomon.main.ApplicationShutdown;

/**
 * @author Vladimir Gordiychuk
 */
@Component
public class StartupShutdownListener implements ApplicationShutdown {
    private static final Logger logger = LoggerFactory.getLogger(StartupShutdownListener.class);

    private final AlertingLocalShards localShards;
    private final EvaluationAssignmentService evaluationAssignmentService;
    private final GrpcEvaluationService grpcEvaluationService;
    private final UnrollExecutor unrollExecutor;
    private final AlertingBalancer balancer;

    @Autowired
    public StartupShutdownListener(
        AlertingLocalShards localShards,
        GrpcEvaluationService grpcEvaluationService,
        EvaluationAssignmentService evaluationAssignmentService,
        UnrollExecutor unrollExecutor,
        AlertingBalancer balancer)
    {
        this.localShards = localShards;
        this.grpcEvaluationService = grpcEvaluationService;
        this.evaluationAssignmentService = evaluationAssignmentService;
        this.unrollExecutor = unrollExecutor;
        this.balancer = balancer;
    }

    private CompletableFuture<Void> safeStop(Runnable task) {
        return CompletableFuture.runAsync(task)
            .exceptionally(e -> {
                logger.error("failed stop " + task, e);
                return null;
            });
    }

    public CompletableFuture<Void> safeStop(Supplier<CompletableFuture<?>> supplier) {
        return CompletableFuture.completedFuture(null)
            .thenCompose(ignore -> supplier.get())
            .handle((o, e) -> {
                if (e != null) {
                    logger.error("failed stop " + supplier, e);
                }
                return null;
            });
    }

    @Override
    public CompletableFuture<Void> shutdown() {
        return CompletableFuture.allOf(
                safeStop(evaluationAssignmentService::close),
                safeStop(grpcEvaluationService::asyncClose),
                safeStop(unrollExecutor::close),
                safeStop(localShards::gracefulShutdown),
                safeStop(balancer::gracefulShutdown));
    }
}
