package ru.yandex.solomon.alert.inject.spring.notification;

import java.io.IOException;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.WillNotClose;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.notification.TemplateVarsFactory;
import ru.yandex.solomon.alert.notification.channel.email.AsyncEmailClient;
import ru.yandex.solomon.alert.notification.channel.email.EmailNotificationChannelFactory;
import ru.yandex.solomon.alert.notification.channel.email.JavamailTransport;
import ru.yandex.solomon.alert.notification.channel.email.MailTransport;
import ru.yandex.solomon.alert.template.TemplateFactory;
import ru.yandex.solomon.config.protobuf.Time;
import ru.yandex.solomon.config.protobuf.TimeUnit;
import ru.yandex.solomon.config.protobuf.alert.BrokerConfig;
import ru.yandex.solomon.config.protobuf.alert.EmailChannelConfig;
import ru.yandex.solomon.config.protobuf.mail.MailClientConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;

import static ru.yandex.solomon.config.OptionalSet.setInt;
import static ru.yandex.solomon.config.OptionalSet.setString;
import static ru.yandex.solomon.config.OptionalSet.setTimeOrDefaultMillis;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
public class EmailContext {
    private final EmailChannelConfig emailConfig;
    @WillNotClose
    private final ScheduledExecutorService notifyExecutor;

    public EmailContext(BrokerConfig config, ThreadPoolProvider threadPool) {
        this.emailConfig = config
                .getNotificationConfig()
                .getEmailConfig();
        this.notifyExecutor = threadPool.getSchedulerExecutorService();
    }

    @Bean
    public AsyncEmailClient emailClient(MailTransport transport) {
        return new AsyncEmailClient(transport);
    }

    @Bean
    public MailTransport mailSender(ThreadPoolProvider threadPool, MetricRegistry registry) {
        MailClientConfig config = emailConfig.getMailClientConfig();

        Time defaultTimeout = Time.newBuilder()
                .setUnit(TimeUnit.SECONDS)
                .setValue(10)
                .build();

        // https://javaee.github.io/javamail/docs/api/com/sun/mail/smtp/package-summary.html
        Properties opts = new Properties();
        opts.setProperty("mail.transport.protocol", "smtp");
        setString(host -> opts.put("mail.smtp.host", host), config.getSmtpHost());
        setInt(port -> opts.put("mail.smtp.port", String.valueOf(port)), config.getSmtpPort());
        setTimeOrDefaultMillis(
                millis -> opts.put("mail.smtp.connectiontimeout", String.valueOf(millis)),
                config.getSmptConnectTimeout(),
                defaultTimeout);
        setTimeOrDefaultMillis(
                millis -> opts.put("mail.smtp.timeout", String.valueOf(millis)),
                config.getSmptReadTimeout(),
                defaultTimeout);
        setTimeOrDefaultMillis(
                millis -> opts.put("mail.smtp.writetimeout", String.valueOf(millis)),
                config.getSmptWriteTimeout(),
                defaultTimeout);

        final int queueCapacity = emailConfig.getSendQueueSize() != 0
                ? emailConfig.getSendQueueSize()
                : 1000;

        final int connections = emailConfig.getConnectionCount() != 0
                ? emailConfig.getConnectionCount()
                : 10;

        ExecutorService executorService =
                threadPool.getExecutorService(
                        emailConfig.getThreadPoolName(),
                        "EmailChannelConfig.ThreadPoolName");

        return new JavamailTransport(opts, connections, queueCapacity, executorService, registry);
    }

    @Bean
    public EmailNotificationChannelFactory emailNotificationChannelFactory(
            TemplateFactory templateFactory,
            AsyncEmailClient emailClient,
            TemplateVarsFactory varsFactory) throws IOException
    {
        EmailTemplate template = EmailTemplate.create(templateFactory, emailConfig);

        return new EmailNotificationChannelFactory(
                emailClient,
                notifyExecutor,
                templateFactory,
                varsFactory,
                template);
    }
}
