package ru.yandex.solomon.alert.inject.spring.notification;

import java.io.IOException;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.notification.domain.email.EmailNotification;
import ru.yandex.solomon.alert.template.Template;
import ru.yandex.solomon.alert.template.TemplateFactory;
import ru.yandex.solomon.config.protobuf.alert.EmailChannelConfig;

import static ru.yandex.solomon.alert.inject.spring.TemplateUtils.makeTemplate;

/**
 * @author Vladimir Gordiychuk
 */
public class EmailTemplate {
    private static final String EMAIL_DEFAULT_SUBJECT_TEMPLATE = "/ru/yandex/solomon/alert/notification/channel/template/email/defaultSubjectTemplate.mustache";
    private static final String EMAIL_DEFAULT_MULTI_ALERT_SUBJECT_TEMPLATE = "/ru/yandex/solomon/alert/notification/channel/template/email/defaultMultiAlertSubjectTemplate.mustache";
    private static final String EMAIL_DEFAULT_CONTENT_TEMPLATE = "/ru/yandex/solomon/alert/notification/channel/template/email/defaultContentTemplate.mustache";
    private static final String EMAIL_DEFAULT_MULTI_ALERT_CONTENT_TEMPLATE = "/ru/yandex/solomon/alert/notification/channel/template/email/defaultMultiAlertContentTemplate.mustache";

    private final String from;

    private final Template subject;
    private final Template content;

    private final Template multipleSubject;
    private final Template multipleContent;

    public EmailTemplate(String from, Template subject, Template content, Template multipleSubject, Template multipleContent) {
        this.from = from;
        this.subject = subject;
        this.content = content;
        this.multipleSubject = multipleSubject;
        this.multipleContent = multipleContent;
    }

    public String getFrom() {
        return from;
    }

    public String getSubject(Map<String, Object> params) {
        return subject.process(params);
    }

    public String getContent(Map<String, Object> params) {
        return content.process(params);
    }

    public String getMultiAlertSubject(Map<String, Object> params) {
        return multipleSubject.process(params);
    }

    public String getMultiAlertContent(Map<String, Object> params) {
        return multipleContent.process(params);
    }

    public EmailTemplate override(TemplateFactory templateFactory, EmailNotification target) {
        if (!isCustomTemplateDefined(target)) {
            return this;
        }

        Template subject = StringUtils.isEmpty(target.getSubjectTemplate())
                ? this.subject
                : templateFactory.createTemplate(target.getSubjectTemplate());

        Template content = StringUtils.isEmpty(target.getContentTemplate())
                ? this.content
                : templateFactory.createTemplate(target.getContentTemplate());

        return new EmailTemplate(from, subject, content, multipleSubject, multipleContent);
    }

    private boolean isCustomTemplateDefined(EmailNotification def) {
        if (StringUtils.isNotEmpty(def.getContentTemplate())) {
            return true;
        }

        if (StringUtils.isNotEmpty(def.getSubjectTemplate())) {
            return true;
        }

        return false;
    }

    public static EmailTemplate create(TemplateFactory factory) throws IOException {
        return create(factory, EmailChannelConfig.getDefaultInstance());
    }

    public static EmailTemplate create(TemplateFactory factory, EmailChannelConfig config) throws IOException {
        final Template subject = makeTemplate(factory, config.getPathToSubjectTemplate(), EMAIL_DEFAULT_SUBJECT_TEMPLATE);
        final Template content = makeTemplate(factory, config.getPathToContentTemplate(), EMAIL_DEFAULT_CONTENT_TEMPLATE);

        final Template mSubject = makeTemplate(factory, config.getPathToMultiAlertSubjectTemplate(), EMAIL_DEFAULT_MULTI_ALERT_SUBJECT_TEMPLATE);
        final Template mContent = makeTemplate(factory, config.getPathToMultiAlertContentTemplate(), EMAIL_DEFAULT_MULTI_ALERT_CONTENT_TEMPLATE);
        final String from = StringUtils.isEmpty(config.getFrom())
                ? "solomon@yandex-team.ru"
                : config.getFrom();

        return new EmailTemplate(from, subject, content, mSubject, mContent);
    }
}
