package ru.yandex.solomon.alert.inject.spring.notification;

import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.jns.client.HttpJnsClient;
import ru.yandex.jns.client.JnsClient;
import ru.yandex.jns.config.JnsClientOptions;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.passport.tvmauth.NativeTvmClient;
import ru.yandex.passport.tvmauth.TvmApiSettings;
import ru.yandex.passport.tvmauth.TvmClient;
import ru.yandex.solomon.alert.notification.channel.phone.PhoneNotificationChannelFactory;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.config.protobuf.alert.BrokerConfig;
import ru.yandex.solomon.config.protobuf.alert.JNSChannelConfig;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.secrets.SecretProvider;

/**
 * @author Alexey Trushkin
 */
@Configuration
@ParametersAreNonnullByDefault
public class PhoneContext {
    private final JNSChannelConfig config;

    public PhoneContext(BrokerConfig config) {
        this.config = config.getNotificationConfig().getJNSChannelConfig();
    }

    @Bean(name = "jnsTvmClient")
    TvmClient jnsTvmClient(TAuthConfig authConfig, SecretProvider secrets) {
        TAuthConfig.TTvmConfig tvmConfig = authConfig.getTvmConfig();
        Optional<String> clientSecret = secrets.getSecret(tvmConfig.getSecret());
        if (tvmConfig.getSecret().isEmpty()
                || clientSecret.isEmpty()
                || config == null
                || config == JNSChannelConfig.getDefaultInstance())
        {
            return null;
        }
        return new NativeTvmClient(TvmApiSettings.create()
                .setSelfTvmId(tvmConfig.getClientId())
                .enableServiceTicketChecking()
                .enableServiceTicketsFetchOptions(clientSecret.get(), Map.of("jns", config.getDestinationTvmId())));
    }

    @Bean
    public JnsClient jnsClient(
            ThreadPoolProvider threadPoolProvider,
            @Qualifier("jnsTvmClient") Optional<TvmClient> tvmClient,
            MetricRegistry metricRegistry)
    {
        if (tvmClient.isEmpty() || config == null) {
            return null;
        }
        JnsClientOptions opts = JnsClientOptions.newBuilder()
                .setUrl(config.getUrl())
                .setMetricRegistry(metricRegistry)
                .setExecutor(threadPoolProvider.getExecutorService("CpuLowPriority", ""))
                .setTokenProvider(() ->
                        AuthType.TvmService.getValuePrefix() + tvmClient.get().getServiceTicketFor(config.getDestinationTvmId()))
                .setTokenHeaderProvider(AuthType.TvmService::getHeaderName)
                .build();
        return new HttpJnsClient(opts);
    }

    @Bean
    public PhoneNotificationChannelFactory phoneNotificationChannelFactory(Optional<JnsClient> jnsClient) {
        if (jnsClient.isEmpty() || config == null) {
            return null;
        }
        return new PhoneNotificationChannelFactory(jnsClient.get(), config);
    }
}
