package ru.yandex.solomon.alert.inject.spring.notification;

import java.io.IOException;
import java.util.Optional;
import java.util.concurrent.ExecutorService;

import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;

import ru.yandex.solomon.alert.charts.ChartsClient;
import ru.yandex.solomon.alert.client.MuteApi;
import ru.yandex.solomon.alert.dao.TelegramDao;
import ru.yandex.solomon.alert.dao.TelegramEventsDao;
import ru.yandex.solomon.alert.notification.TemplateVarsFactory;
import ru.yandex.solomon.alert.notification.channel.telegram.ChatIdResolverImpl;
import ru.yandex.solomon.alert.notification.channel.telegram.ChatIdStorage;
import ru.yandex.solomon.alert.notification.channel.telegram.RawTelegramLimits;
import ru.yandex.solomon.alert.notification.channel.telegram.TelegramEventsCleaner;
import ru.yandex.solomon.alert.notification.channel.telegram.TelegramNotificationChannelFactory;
import ru.yandex.solomon.alert.notification.channel.telegram.TelegramTemplate;
import ru.yandex.solomon.alert.notification.channel.telegram.TelegramUpdatesReceiver;
import ru.yandex.solomon.alert.telegram.DefaultTelegramClient;
import ru.yandex.solomon.alert.telegram.TelegramClient;
import ru.yandex.solomon.alert.template.TemplateFactory;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.alert.BrokerConfig;
import ru.yandex.solomon.config.protobuf.alert.TelegramChannelConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.util.SolomonEnv;
import ru.yandex.staff.StaffClient;


/**
 * @author alexlovkov
 **/
@Configuration
public class TelegramContext {
    private static final String TELEGRAM_API = "https://api.telegram.org";

    private final TelegramChannelConfig config;
    private final ThreadPoolProvider threadPoolProvider;
    private final SecretProvider secretProvider;

    public TelegramContext(BrokerConfig config, ThreadPoolProvider threadPoolProvider, SecretProvider secretProvider) {
        this.config = config.getNotificationConfig().getTelegramConfig();
        this.threadPoolProvider = threadPoolProvider;
        this.secretProvider = secretProvider;
    }

    @Bean
    @Conditional(TelegramCondition.class)
    public TelegramClient telegramClient() {
        ExecutorService executor = threadPoolProvider
                .getExecutorService(config.getThreadPoolName(), "BrokerConfig.NotificationConfig.TelegramConfig");

        Optional<String> token = secretProvider.getSecret(config.getToken());
        if (SolomonEnv.DEVELOPMENT.isActive() && token.isEmpty()) {
            // TODO: create stub client
            return new DefaultTelegramClient(TELEGRAM_API, "", executor);
        }

        return new DefaultTelegramClient(
                TELEGRAM_API,
                token.orElseThrow(() -> new RuntimeException("cannot create TelegramClient without token")),
                executor);
    }

    @Bean
    public ChatIdStorage chatIdResolver(TelegramDao dao) {
        return new ChatIdResolverImpl(dao);
    }

    @Bean
    @Conditional(TelegramCondition.class)
    public TelegramNotificationChannelFactory telegramNotificationChannelFactory(
        TelegramClient client,
        StaffClient staffClient,
        TemplateFactory templateFactory,
        ThreadPoolProvider threadPoolProvider,
        ChatIdStorage chatIdResolver,
        ChartsClient chartsClient,
        TelegramEventsDao telegramEventsDao,
        TemplateVarsFactory vars) throws IOException
    {
        TelegramTemplate template = TelegramTemplate.create(templateFactory, vars);

        int generalRateLimit = config.getRateLimitEventsPerSecond() > 0
            ? config.getRateLimitEventsPerSecond()
            : 30;
        int rateLimitPerChat = config.getRateLimitEventsPerChat() > 0
            ? config.getRateLimitEventsPerChat()
            : 1;
        double groupRateLimit = config.getRateLimitEventsPerSecondForGroup() > 0
            ? config.getRateLimitEventsPerSecondForGroup()
            : 0.3;
        RawTelegramLimits rawLimits = new RawTelegramLimits(generalRateLimit, rateLimitPerChat, groupRateLimit);
        return new TelegramNotificationChannelFactory(client, staffClient, chartsClient, chatIdResolver,
            templateFactory, template, rawLimits,
            threadPoolProvider.getSchedulerExecutorService(), telegramEventsDao);
    }

    @Bean
    @Conditional(TelegramCondition.class)
    public TelegramUpdatesReceiver telegramUpdatesReceiver(
        TelegramClient client,
        ThreadPoolProvider threadPoolProvider,
        TelegramDao dao,
        StaffClient staffClient,
        DistributedLock distributedLock,
        ChatIdStorage chatIdResolver,
        TelegramEventsDao telegramEventsDao,
        ChartsClient chartsClient,
        TemplateVarsFactory templateVarsFactory,
        MuteApi muteApi)
    {
        ExecutorService executor = threadPoolProvider.getExecutorService(
            config.getThreadPoolName(), "BrokerConfig.NotificationConfig.TelegramConfig");
        return new TelegramUpdatesReceiver(client, threadPoolProvider.getSchedulerExecutorService(),
            executor, config, dao, staffClient, distributedLock, chatIdResolver, telegramEventsDao, chartsClient,
            templateVarsFactory, muteApi);
    }

    @Bean
    @Conditional(TelegramCondition.class)
    public TelegramEventsCleaner telegramEventsCleaner(
        TelegramEventsDao telegramEventsDao,
        ThreadPoolProvider threadPoolProvider)
    {
        return new TelegramEventsCleaner(
            telegramEventsDao,
            TimeConverter.protoToDuration(config.getTtlDeletion()),
            threadPoolProvider.getSchedulerExecutorService());
    }
}
