package ru.yandex.solomon.alert.inject.spring.persistence;

import java.time.Clock;
import java.util.Optional;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.yandex.ydb.core.rpc.RpcTransport;
import com.yandex.ydb.table.SchemeClient;
import com.yandex.ydb.table.TableClient;
import com.yandex.ydb.table.rpc.grpc.GrpcTableRpc;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.dao.AlertStatesDao;
import ru.yandex.solomon.alert.dao.AlertTemplateDao;
import ru.yandex.solomon.alert.dao.AlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.dao.AlertsDao;
import ru.yandex.solomon.alert.dao.CachedAlertTemplateDao;
import ru.yandex.solomon.alert.dao.MutesDao;
import ru.yandex.solomon.alert.dao.NotificationsDao;
import ru.yandex.solomon.alert.dao.SchemaAwareLocksDao;
import ru.yandex.solomon.alert.dao.ShardsDao;
import ru.yandex.solomon.alert.dao.TelegramDao;
import ru.yandex.solomon.alert.dao.TelegramEventsDao;
import ru.yandex.solomon.alert.dao.ydb.YdbAlertStatesDaoFactory;
import ru.yandex.solomon.alert.dao.ydb.YdbAlertTemplateDao;
import ru.yandex.solomon.alert.dao.ydb.YdbAlertTemplateLastVersionDao;
import ru.yandex.solomon.alert.dao.ydb.YdbAlertsDaoFactory;
import ru.yandex.solomon.alert.dao.ydb.YdbLocksDaoFactory;
import ru.yandex.solomon.alert.dao.ydb.YdbMutesDaoFactory;
import ru.yandex.solomon.alert.dao.ydb.YdbNotificationsDaoFactory;
import ru.yandex.solomon.alert.dao.ydb.YdbSchemaVersion;
import ru.yandex.solomon.alert.dao.ydb.YdbTelegramDaoFactory;
import ru.yandex.solomon.alert.dao.ydb.YdbTelegramEventsDaoFactory;
import ru.yandex.solomon.alert.dao.ydb.entity.YdbNotificationsDao;
import ru.yandex.solomon.alert.dao.ydb.entity.YdbShardsDao;
import ru.yandex.solomon.balancer.dao.BalancerDao;
import ru.yandex.solomon.balancer.dao.YdbBalancerDaoV2;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.core.db.YdbClientsContext;
import ru.yandex.solomon.core.db.dao.ClusterFlagsDao;
import ru.yandex.solomon.core.db.dao.ProjectFlagsDao;
import ru.yandex.solomon.core.db.dao.ProjectsDao;
import ru.yandex.solomon.core.db.dao.QuotasDao;
import ru.yandex.solomon.core.db.dao.ServiceFlagsDao;
import ru.yandex.solomon.core.db.dao.ServiceProvidersDao;
import ru.yandex.solomon.core.db.dao.ShardFlagsDao;
import ru.yandex.solomon.core.db.dao.client.GrpcProjectManagerLegacyClient;
import ru.yandex.solomon.core.db.dao.client.YdbOrClientProjectsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbClusterFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbProjectsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbQuotasDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServiceFlagsDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbServiceProvidersDao;
import ru.yandex.solomon.core.db.dao.ydb.YdbShardFlagsDao;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.selfmon.mon.DaoMetricsProxy;
import ru.yandex.solomon.ydb.YdbTableClientMetrics;

import static ru.yandex.solomon.core.db.dao.ConfigDaoContext.CLUSTER_FLAGS_TABLE;
import static ru.yandex.solomon.core.db.dao.ConfigDaoContext.PROJECT_FLAGS_TABLE;
import static ru.yandex.solomon.core.db.dao.ConfigDaoContext.SERVICE_FLAGS_TABLE;
import static ru.yandex.solomon.core.db.dao.ConfigDaoContext.SHARD_FLAGS_TABLE;

/**
 * @author Vladimir Gordiychuk
 */
@Configuration
@Import(YdbClientsContext.class)
public class YdbPersistenceContext {
    private final ObjectMapper objectMapper;
    private final String root;
    private final MetricRegistry registry;

    private final TableClient table;
    private final SchemeClient scheme;
    private final FeatureFlagsHolder featureFlagsHolder;

    public YdbPersistenceContext(
        @Qualifier("configKikimrRootPath") String root,
        @Qualifier("globalTableClient") TableClient globalTableClient,
        @Qualifier("globalSchemeClient") SchemeClient globalSchemeClient,
        MetricRegistry registry,
        FeatureFlagsHolder featureFlagsHolder)
    {
        this.featureFlagsHolder = featureFlagsHolder;
        this.objectMapper = new ObjectMapper();
        this.root = root;
        this.table = globalTableClient;
        this.scheme = globalSchemeClient;
        this.registry = registry;
    }

    @Bean
    public YdbSchemaVersion ydbSchemaVersion() {
        registry.lazyGaugeInt64("ydb.schema.version", Labels.of("current", YdbSchemaVersion.CURRENT.name()), () -> 1);
        registry.lazyGaugeInt64("ydb.schema.version", Labels.of("min", YdbSchemaVersion.MIN.name()), () -> 1);
        return YdbSchemaVersion.CURRENT;
    }

    @Bean("locksTableClient")
    public TableClient locksTableClient(@Qualifier("globalTransport") RpcTransport transport) {
        return TableClient.newClient(GrpcTableRpc.useTransport(transport))
                .queryCacheSize(100)
                .sessionPoolSize(5, 10)
                .build();
    }

    @Bean(name = "locksClientMetrics")
    MetricSupplier tableClientMetrics(@Qualifier("locksTableClient") TableClient table) {
        return new YdbTableClientMetrics(table, "locks");
    }

    @Bean
    public SchemaAwareLocksDao locksDao(Clock clock, @Qualifier("locksTableClient") TableClient table, YdbSchemaVersion version) {
        var dao = YdbLocksDaoFactory.create(root, table, scheme, version, clock);
        return measure(dao, SchemaAwareLocksDao.class);
    }

    @Bean
    public TelegramDao telegramDao(YdbSchemaVersion version) {
        var dao = YdbTelegramDaoFactory.create(root, table, scheme, version);
        return measure(dao, TelegramDao.class);
    }

    @Bean
    public TelegramEventsDao telegramEventsDao(YdbSchemaVersion version) {
        var dao = YdbTelegramEventsDaoFactory.create(root, table, scheme, version);
        return measure(dao, TelegramEventsDao.class);
    }

    @Bean
    public AlertsDao alertsDao(YdbSchemaVersion version) {
        var dao = YdbAlertsDaoFactory.create(root, table, scheme, version, objectMapper);
        return measure(dao, AlertsDao.class);
    }

    @Bean
    public NotificationsDao notificationsDao(YdbSchemaVersion version) {
        var dao = YdbNotificationsDaoFactory.create(root, table, scheme, version, objectMapper);
        ((YdbNotificationsDao)dao).migrateSchema();
        return measure(dao, NotificationsDao.class);
    }

    @Bean
    public MutesDao mutesDao(YdbSchemaVersion version) {
        var dao = YdbMutesDaoFactory.create(root, table, scheme, version, objectMapper);
        return measure(dao, MutesDao.class);
    }

    @Bean
    public AlertStatesDao alertStatesDao(YdbSchemaVersion version) {
        var dao = YdbAlertStatesDaoFactory.create(root, table, scheme, version, registry);
        return measure(dao, AlertStatesDao.class);
    }

    @Bean
    public BalancerDao alertingBalancerDao(YdbSchemaVersion version) {
        var rootPath = root + "/Alerting/" + version.folderName();
        var v2 = new YdbBalancerDaoV2(rootPath, table, scheme);
        v2.createSchema();
        return measure(v2, BalancerDao.class);
    }

    @Bean
    public ShardsDao shardsDao(YdbSchemaVersion version) {
        var ydb = new YdbShardsDao(root, table, scheme, version);
        return measure(ydb, ShardsDao.class);
    }

    @Bean
    public ProjectsDao projectsDao(Optional<GrpcProjectManagerLegacyClient> projectManagerClient, ThreadPoolProvider threads) {
        var executor = threads.getExecutorService("CpuLowPriority", "");
        var dao = new YdbProjectsDao(table, root + "/Config/V2/Project", objectMapper, executor);
        var result = new YdbOrClientProjectsDao(featureFlagsHolder, dao, projectManagerClient);
        return measure(result, ProjectsDao.class);
    }

    @Bean
    public ServiceProvidersDao serviceProvidersDao() {
        var dao = new YdbServiceProvidersDao(table, root + "/Config/V2/ServiceProvider", objectMapper);
        dao.migrateSchema();
        return measure(dao, ServiceProvidersDao.class);
    }

    @Bean
    public QuotasDao quotasDao() {
        var dao = new YdbQuotasDao(table, root + "/Config/V2/Quota");
        return measure(dao, QuotasDao.class);
    }

    @Bean
    public ProjectFlagsDao projectFlagsDao() {
        var tablePath = root + "/Config/V2/" + PROJECT_FLAGS_TABLE;
        var dao = new YdbProjectFlagsDao(table, tablePath);
        return measure(dao, ProjectFlagsDao.class);
    }

    @Bean
    public ServiceFlagsDao serviceFlagsDao() {
        var tablePath = root + "/Config/V2/" + SERVICE_FLAGS_TABLE;
        var dao = new YdbServiceFlagsDao(table, tablePath);
        return measure(dao, ServiceFlagsDao.class);
    }

    @Bean
    public ClusterFlagsDao clusterFlagsDao() {
        var tablePath = root + "/Config/V2/" + CLUSTER_FLAGS_TABLE;
        var dao = new YdbClusterFlagsDao(table, tablePath);
        return measure(dao, ClusterFlagsDao.class);
    }

    @Bean
    public ShardFlagsDao shardFlagsDao() {
        var tablePath = root + "/Config/V2/" + SHARD_FLAGS_TABLE;
        var dao = new YdbShardFlagsDao(table, tablePath);
        return measure(dao, ShardFlagsDao.class);
    }

    @Bean
    public AlertTemplateDao cachedAlertTemplateDao(YdbSchemaVersion version) {
        var dao = new YdbAlertTemplateDao(root, table, scheme, version, objectMapper);
        return new CachedAlertTemplateDao(measure(dao, AlertTemplateDao.class));
    }

    @Bean
    public AlertTemplateLastVersionDao alertTemplateLastVersionDao(YdbSchemaVersion version) {
        var dao = new YdbAlertTemplateLastVersionDao(root, table, scheme, version);
        return measure(dao, AlertTemplateLastVersionDao.class);
    }

    private <TDao> TDao measure(TDao dao, Class<TDao> daoClass) {
        return DaoMetricsProxy.of(dao, daoClass, registry);
    }
}
