package ru.yandex.solomon.alert.inject.spring.rest;

import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import ru.yandex.monlib.metrics.CompositeMetricSupplier;
import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.encode.MetricFormat;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.cluster.balancer.AlertingLocalShards;
import ru.yandex.solomon.selfmon.mon.HttpMetricRegistryEncoder;

/**
 * @author Vladimir Gordiychuk
 */
@RestController
public class AlertingMetricsController {
    private static final Logger logger = LoggerFactory.getLogger(AlertingMetricsController.class);
    private final MetricSupplier metricSupplier;
    private final AlertingLocalShards localShards;
    private final ScheduledExecutorService executorService;
    private volatile ResponseEntity<byte[]> latestAlertStatus =
            ResponseEntity.status(HttpStatus.NO_CONTENT)
                    .body(new byte[0]);

    public AlertingMetricsController(List<MetricSupplier> metrics,
                                     AlertingLocalShards localShards,
                                     ScheduledExecutorService executorService)
    {
        this.metricSupplier = new CompositeMetricSupplier(metrics);
        this.localShards = localShards;
        this.executorService = executorService;
        this.executorService.schedule(this::runScheduledRefresh, 1, TimeUnit.MINUTES);
    }

    @RequestMapping({ "/metrics" })
    public ResponseEntity<byte[]> metrics(ServerHttpRequest request) {
        return HttpMetricRegistryEncoder.encode(metricSupplier, request);
    }

    @RequestMapping({ "/metrics/alertsStatus" })
    public ResponseEntity<byte[]> alertsStatus() {
        return latestAlertStatus;
    }

    private void runScheduledRefresh() {
        try {
            refreshAlertStatus();
        } catch (Throwable e) {
            logger.error("Failed refresh alert status", e);
        }

        executorService.schedule(this::runScheduledRefresh, 30, TimeUnit.SECONDS);
    }

    private void refreshAlertStatus() {
        MetricSupplier provider = new MetricSupplier() {
            @Override
            public int estimateCount() {
                return 0;
            }

            @Override
            public void supply(long tsMillis, MetricConsumer consumer) {
                consumer.onStreamBegin(-1);
                consumer.onLabelsBegin(1);
                consumer.onLabel("host", "");
                consumer.onLabelsEnd();
                append(0, Labels.empty(), consumer);
                consumer.onStreamEnd();
            }

            @Override
            public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
                for (var shard : localShards) {
                    shard.appendMetrics(consumer);
                }
            }
        };

        latestAlertStatus = HttpMetricRegistryEncoder.encode(provider, MetricFormat.SPACK, CompressionAlg.ZSTD);
    }
}
