package ru.yandex.solomon.alert.mute.domain;

import java.time.Instant;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.dao.Entity;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public abstract class Mute implements Entity {
    private static final long ARCHIVE_MILLIS = TimeUnit.DAYS.toMillis(1);
    private static final long TTL_MILLIS = TimeUnit.DAYS.toMillis(7);

    private final String id;
    private final String projectId;
    private final String folderId;

    private final String name;
    private final String description;
    private final String ticketId;

    private final Instant from;
    private final Instant to;

    private final Instant ttlBase;

    private final int version;
    private final String createdBy;
    private final String updatedBy;
    private final Instant createdAt;
    private final Instant updatedAt;

    protected Mute(MuteBuilder<?, ?> builder) {
        id = Objects.requireNonNull(builder.id, "id");
        projectId = Objects.requireNonNull(builder.projectId, "projectId");
        name = Objects.requireNonNull(builder.name, "name");
        folderId = builder.folderId;
        description = builder.description;
        ticketId = builder.ticketId;
        from = Objects.requireNonNull(builder.from, "from");
        to = Objects.requireNonNull(builder.to, "to");
        ttlBase = Objects.requireNonNull(builder.ttlBase, "ttlBase");
        version = builder.version;
        createdBy = builder.createdBy;
        createdAt = Objects.requireNonNull(builder.createdAt, "createdAt");
        updatedBy = builder.updatedBy;
        updatedAt = Objects.requireNonNull(builder.updatedAt, "updatedAt");
    }

    public abstract MuteType getType();

    @Override
    public String getId() {
        return id;
    }

    @Override
    public String getProjectId() {
        return projectId;
    }

    @Override
    public int getVersion() {
        return version;
    }

    @Override
    public String getFolderId() {
        return folderId;
    }

    @Override
    public String getCreatedBy() {
        return createdBy;
    }

    @Override
    public String getUpdatedBy() {
        return updatedBy;
    }

    @Override
    public long getCreatedAt() {
        return createdAt.toEpochMilli();
    }

    @Override
    public long getUpdatedAt() {
        return updatedAt.toEpochMilli();
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    public String getTicketId() {
        return ticketId;
    }

    public Instant getFrom() {
        return from;
    }

    public Instant getTo() {
        return to;
    }

    public Instant getTtlBase() {
        return ttlBase;
    }

    public boolean isDeletedByTtl(Instant at) {
        return isDeletedByTtl(at.toEpochMilli());
    }

    public boolean isDeletedByTtl(long atMillis) {
        return atMillis - TTL_MILLIS >= ttlBase.toEpochMilli();
    }

    public MuteStatus getStatusAt(Instant at) {
        return getStatusAt(at.toEpochMilli());
    }

    public MuteStatus getStatusAt(long atMillis) {
        long toMillis = to.toEpochMilli();
        long fromMillis = from.toEpochMilli();
        if (atMillis - ARCHIVE_MILLIS >= toMillis) {
            return MuteStatus.ARCHIVED;
        }
        if (atMillis >= toMillis) {
            return MuteStatus.EXPIRED;
        }
        if (atMillis < fromMillis) {
            return MuteStatus.PENDING;
        }
        return MuteStatus.ACTIVE;
    }

    public abstract boolean matches(String alertId, Labels subAlertLabels);

    public abstract MuteBuilder<?, ?> toBuilder();

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Mute mute = (Mute) o;
        return version == mute.version &&
                id.equals(mute.id) &&
                projectId.equals(mute.projectId) &&
                folderId.equals(mute.folderId) &&
                name.equals(mute.name) &&
                description.equals(mute.description) &&
                ticketId.equals(mute.ticketId) &&
                from.equals(mute.from) &&
                to.equals(mute.to) &&
                ttlBase.equals(mute.ttlBase) &&
                createdBy.equals(mute.createdBy) &&
                updatedBy.equals(mute.updatedBy) &&
                createdAt.equals(mute.createdAt) &&
                updatedAt.equals(mute.updatedAt);
    }

    @Override
    public int hashCode() {
        int result = id.hashCode();
        result = 31 * result + projectId.hashCode();
        result = 31 * result + folderId.hashCode();
        result = 31 * result + name.hashCode();
        result = 31 * result + description.hashCode();
        result = 31 * result + ticketId.hashCode();
        result = 31 * result + from.hashCode();
        result = 31 * result + to.hashCode();
        result = 31 * result + ttlBase.hashCode();
        result = 31 * result + version;
        result = 31 * result + createdBy.hashCode();
        result = 31 * result + updatedBy.hashCode();
        result = 31 * result + createdAt.hashCode();
        result = 31 * result + updatedAt.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "Mute{" +
                "id='" + id + '\'' +
                ", projectId='" + projectId + '\'' +
                ", folderId='" + folderId + '\'' +
                ", name='" + name + '\'' +
                ", description='" + description + '\'' +
                ", ticketId='" + ticketId + '\'' +
                ", from=" + from +
                ", to=" + to +
                ", ttlBase=" + ttlBase +
                ", version=" + version +
                ", createdBy='" + createdBy + '\'' +
                ", updatedBy='" + updatedBy + '\'' +
                ", createdAt=" + createdAt +
                ", updatedAt=" + updatedAt +
                '}';
    }

    public abstract static class MuteBuilder<Derived extends MuteBuilder<?, ?>, DerivedMute extends Mute> {
        private String id;
        private String projectId;
        private String folderId = "";
        private String name;
        private String description = "";
        private String ticketId = "";
        private Instant from;
        private Instant to;
        private Instant ttlBase;
        private int version = 0;
        private String createdBy = "";
        private Instant createdAt;
        private String updatedBy = "";
        private Instant updatedAt;

        protected MuteBuilder() {
        }

        protected MuteBuilder(Mute mute) {
            id = mute.getId();
            projectId = mute.getProjectId();
            folderId = mute.getFolderId();
            name = mute.getName();
            description = mute.getDescription();
            ticketId = mute.getTicketId();
            from = mute.getFrom();
            to = mute.getTo();
            ttlBase = mute.getTtlBase();
            version = mute.getVersion();
            createdBy = mute.getCreatedBy();
            updatedBy = mute.getUpdatedBy();
            createdAt = Instant.ofEpochMilli(mute.getCreatedAt());
            updatedAt = Instant.ofEpochMilli(mute.getUpdatedAt());
        }

        @SuppressWarnings("unchecked")
        private Derived self() {
            return (Derived) this;
        }

        public String getId() {
            return id;
        }

        public Derived setId(String id) {
            this.id = id;
            return self();
        }

        public String getProjectId() {
            return projectId;
        }

        public Derived setProjectId(String projectId) {
            this.projectId = projectId;
            return self();
        }

        public String getFolderId() {
            return folderId;
        }

        public Derived setFolderId(String folderId) {
            this.folderId = folderId;
            return self();
        }

        public String getName() {
            return name;
        }

        public Derived setName(String name) {
            this.name = name;
            return self();
        }

        public String getDescription() {
            return description;
        }

        public Derived setDescription(String description) {
            this.description = description;
            return self();
        }

        public String getTicketId() {
            return ticketId;
        }

        public Derived setTicketId(String ticketId) {
            this.ticketId = ticketId;
            return self();
        }

        public Instant getFrom() {
            return from;
        }

        public Derived setFrom(Instant from) {
            this.from = from;
            return self();
        }

        public Instant getTo() {
            return to;
        }

        public Derived setTo(Instant to) {
            this.to = to;
            return self();
        }

        public Instant getTtlBase() {
            return ttlBase;
        }

        public Derived setTtlBase(Instant ttlBase) {
            this.ttlBase = ttlBase;
            return self();
        }

        public int getVersion() {
            return version;
        }

        public Derived setVersion(int version) {
            this.version = version;
            return self();
        }

        public String getCreatedBy() {
            return createdBy;
        }

        public Derived setCreatedBy(String createdBy) {
            this.createdBy = createdBy;
            return self();
        }

        public Instant getCreatedAt() {
            return createdAt;
        }

        public Derived setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
            return self();
        }

        public String getUpdatedBy() {
            return updatedBy;
        }

        public Derived setUpdatedBy(String updatedBy) {
            this.updatedBy = updatedBy;
            return self();
        }

        public Instant getUpdatedAt() {
            return updatedAt;
        }

        public Derived setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return self();
        }

        public abstract DerivedMute build();
    }
}
