package ru.yandex.solomon.alert.mute.domain;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.query.Selector;
import ru.yandex.solomon.labels.query.Selectors;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class SelectorsMute extends Mute {
    private final Selector alertSelector;
    private final Selectors labelSelectors;

    private SelectorsMute(Builder builder) {
        super(builder);
        this.alertSelector = Objects.requireNonNull(builder.alertSelector, "alertSelector");
        this.labelSelectors = Objects.requireNonNull(builder.labelSelectors, "labelSelectors");
    }

    public Selector getAlertSelector() {
        return alertSelector;
    }

    public Selectors getLabelSelectors() {
        return labelSelectors;
    }

    @Override
    public MuteType getType() {
        return MuteType.BY_SELECTORS;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        if (!super.equals(o)) {
            return false;
        }
        SelectorsMute that = (SelectorsMute) o;
        return alertSelector.equals(that.alertSelector) && labelSelectors.equals(that.labelSelectors);
    }

    @Override
    public int hashCode() {
        return Objects.hash(super.hashCode(), alertSelector, labelSelectors);
    }

    @Override
    public boolean matches(String alertId, Labels subAlertLabels) {
        if (!alertSelector.match(alertId)) {
            return false;
        }
        return labelSelectors.match(subAlertLabels);
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    @Override
    public String toString() {
        return "SelectorsMute{" +
                "super=" + super.toString() +
                ", alertSelector=" + alertSelector +
                ", labelSelectors=" + labelSelectors +
                '}';
    }

    public static class Builder extends Mute.MuteBuilder<Builder, SelectorsMute> {
        private Selector alertSelector = Selector.any("alert");
        private Selectors labelSelectors = Selectors.of();

        private Builder() {
        }

        private Builder(SelectorsMute selectorsMute) {
            super(selectorsMute);
            alertSelector = selectorsMute.getAlertSelector();
            labelSelectors = selectorsMute.getLabelSelectors();
        }

        public Builder setAlertSelector(Selector alertSelector) {
            this.alertSelector = alertSelector;
            return this;
        }

        public Builder setLabelSelectors(Selectors labelSelectors) {
            this.labelSelectors = labelSelectors;
            return this;
        }

        @Override
        public SelectorsMute build() {
            return new SelectorsMute(this);
        }
    }
}
