package ru.yandex.solomon.alert.notification;

import java.util.EnumMap;
import java.util.EnumSet;
import java.util.concurrent.TimeUnit;

import ru.yandex.monlib.metrics.MetricConsumer;
import ru.yandex.monlib.metrics.MetricSupplier;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;

/**
 * @author Vladimir Gordiychuk
 */
public class ChannelMetrics implements MetricSupplier {
    private static final EnumSet<NotificationStatus.Code> NOT_TRACK_TIME_FOR_STATUS = EnumSet.of(
            NotificationStatus.Code.SKIP_BY_STATUS,
            NotificationStatus.Code.SKIP_REPEAT,
            NotificationStatus.Code.ABSENT_NOTIFICATION_CHANNEL,
            NotificationStatus.Code.MUTED
    );

    private final MetricRegistry registry;

    private final Rate started;
    private final Rate completed;
    private final LazyGaugeInt64 inflight;
    private final Histogram elapsedTime;
    private final Histogram notifyLag;
    private final EnumMap<NotificationStatus.Code, Rate> statusCounter;
    private final Rate retry;

    public ChannelMetrics(Labels commonLabels) {
        // TODO: use flat and not thread safe registry (gordiychuk@)
        registry = new MetricRegistry(commonLabels);
        started = registry.rate("notifications.channel.notify.started");
        completed = registry.rate("notifications.channel.notify.completed");
        inflight = registry.lazyGaugeInt64("notifications.channel.notify.inFlight", () -> this.started.get() - this.completed.get());
        retry = registry.rate("notifications.channel.retry.count");
        elapsedTime = registry.histogramRate("notifications.channel.notify.elapsedTimeMillis", Histograms.exponential(20, 2, 16));
        notifyLag = registry.histogramRate("notifications.channel.notify.lagSeconds", Histograms.exponential(12, 2, 1));

        EnumMap<NotificationStatus.Code, Rate> statusCounter = new EnumMap<>(NotificationStatus.Code.class);
        for (NotificationStatus.Code status : NotificationStatus.Code.values()) {
            Rate counter = registry.rate("notifications.channel.notify.status.total", Labels.of("status", status.name()));
            statusCounter.put(status, counter);
        }

        this.statusCounter = statusCounter;
    }

    public void retry(Event event) {
        this.retry.inc();
    }

    public long started(Event event) {
        started.inc();
        return System.nanoTime();
    }

    public void completed(Event event, NotificationStatus status, long startTime) {
        if (!NOT_TRACK_TIME_FOR_STATUS.contains(status.getCode())) {
            long elapsedTime = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);
            this.elapsedTime.record(elapsedTime);
        }

        completed.inc();
        statusCounter.get(status.getCode()).inc();
        if (status.getCode() == NotificationStatus.Code.SUCCESS) {
            notifyLag.record(TimeUnit.MILLISECONDS.toSeconds(System.currentTimeMillis() - event.getState().getLatestEval().toEpochMilli()));
        }
    }

    void combine(ChannelMetrics metrics) {
        this.started.combine(metrics.started);
        this.completed.combine(metrics.completed);
        this.retry.combine(metrics.retry);
        this.elapsedTime.record(metrics.elapsedTime.snapshot());
        this.notifyLag.record(metrics.notifyLag.snapshot());
        this.statusCounter.forEach((code, counter) -> counter.combine(metrics.statusCounter.get(code)));
    }

    @Override
    public int estimateCount() {
        return registry.estimateCount();
    }

    @Override
    public void append(long tsMillis, Labels commonLabels, MetricConsumer consumer) {
        registry.append(tsMillis, commonLabels, consumer);
    }
}
