package ru.yandex.solomon.alert.notification;

import java.time.Instant;
import java.util.EnumSet;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.base.MoreObjects;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.rule.EvaluationState;

import static java.util.Objects.requireNonNull;

/**
 * @author Vladimir Gordiychuk
 */
@Immutable
@ThreadSafe
@ParametersAreNonnullByDefault
public final class NotificationState {
    // These codes are considered as final and will not be resend on the next evaluation
    private static final Set<NotificationStatus.Code> SET_LATEST_SUCCESS_ON_CODES = EnumSet.of(
            NotificationStatus.Code.SUCCESS,
            NotificationStatus.Code.INVALID_REQUEST,
            NotificationStatus.Code.NOT_SUBSCRIBED,
            NotificationStatus.Code.PERMISSION_DENIED
    );
    private final NotificationKey key;
    private final NotificationStatus latestStatus;
    private final Instant latestEval;
    private final Instant latestSuccessNotify;
    @Nullable
    private final EvaluationStatus.Code latestNotifiedEvalStatusCode;

    private NotificationState(Builder builder) {
        this.key = requireNonNull(builder.key);
        this.latestStatus = builder.latestStatus != null
                ? builder.latestStatus
                : NotificationStatus.OBSOLETE;
        this.latestEval = builder.latestEval != null
                ? builder.latestEval
                : Instant.EPOCH;
        this.latestSuccessNotify = builder.latestSuccessNotify != null
                ? builder.latestSuccessNotify
                : Instant.EPOCH;
        this.latestNotifiedEvalStatusCode = builder.latestNotifiedEvalStatusCode;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static NotificationState init(NotificationKey key) {
        return newBuilder()
                .setKey(key)
                .build();
    }

    public NotificationState nextStatus(NotificationStatus status, EvaluationState evaluationState) {
        var builder = toBuilder()
                .setLatestEval(evaluationState.getLatestEval())
                .setLatestStatus(status);

        if (SET_LATEST_SUCCESS_ON_CODES.contains(status.getCode())) {
            builder.setLatestSuccessNotify(evaluationState.getLatestEval());
            builder.setLatestNotifiedEvalStatusCode(evaluationState.getStatus().getCode());
        }

        return builder.build();
    }

    public Builder toBuilder() {
        return new Builder(this);
    }

    public NotificationKey getKey() {
        return key;
    }

    public NotificationStatus getLatestStatus() {
        return latestStatus;
    }

    public Instant getLatestEval() {
        return latestEval;
    }

    public Instant getLatestSuccessNotify() {
        return latestSuccessNotify;
    }

    @Nullable
    public EvaluationStatus.Code getLatestNotifiedEvalStatusCode() {
        return latestNotifiedEvalStatusCode;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .omitNullValues()
                .add("key", key)
                .add("latestStatus", latestStatus)
                .add("latestEval", latestEval)
                .add("latestSuccessNotify", latestSuccessNotify)
                .add("latestNotifiedEvalStatusCode", latestNotifiedEvalStatusCode)
                .toString();
    }

    public static class Builder {
        private NotificationKey key;
        private NotificationStatus latestStatus;
        private Instant latestEval;
        private Instant latestSuccessNotify;
        private EvaluationStatus.Code latestNotifiedEvalStatusCode;


        private Builder() {
        }

        private Builder(NotificationState state) {
            this.key = state.key;
            this.latestStatus = state.latestStatus;
            this.latestEval = state.latestEval;
            this.latestSuccessNotify = state.latestSuccessNotify;
            this.latestNotifiedEvalStatusCode = state.latestNotifiedEvalStatusCode;
        }

        public Builder setKey(NotificationKey key) {
            this.key = key;
            return this;
        }

        public Builder setLatestStatus(NotificationStatus latestStatus) {
            this.latestStatus = latestStatus;
            return this;
        }

        public Builder setLatestEval(Instant latestEval) {
            this.latestEval = latestEval;
            return this;
        }

        public Builder setLatestSuccessNotify(Instant latestSuccessNotify) {
            this.latestSuccessNotify = latestSuccessNotify;
            return this;
        }

        public Builder setLatestNotifiedEvalStatusCode(EvaluationStatus.Code latestNotifiedEvalStatusCode) {
            this.latestNotifiedEvalStatusCode = latestNotifiedEvalStatusCode;
            return this;
        }

        public NotificationState build() {
            return new NotificationState(this);
        }
    }
}

