package ru.yandex.solomon.alert.notification;

import java.util.concurrent.TimeUnit;

import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.NotThreadSafe;
import javax.annotation.concurrent.ThreadSafe;

import com.google.common.base.MoreObjects;

/**
 * @author Vladimir Gordiychuk
 */
// TODO: need generic retry options definition for whole solomon (gordiychuk@)
@Immutable
@ThreadSafe
public final class RetryOptions {
    private static final long DEFAULT_INITIAL_DELAY_MILLIS = TimeUnit.MILLISECONDS.toMillis(100);
    private static final double DEFAULT_RETRY_DELAY_MULTIPLIER = 1;
    private static final long DEFAULT_MAX_RETRY_DELAY_MILLIS = TimeUnit.MINUTES.toMillis(5);

    private static final RetryOptions EMPTY = RetryOptions.newBuilder().build();

    /**
     * The number of retries allowed before a task can fail permanently.
     * If both taskRetryLimit and taskAgeLimitMillis are specified, then one of
     * limits must be exceeded before a task can fail permanently.
     */
    private final int taskRetryLimit;

    /**
     * The maximum age from the first attempt to execute a task after which
     * any new task failure can be permanent.
     * If both taskRetryLimit and taskAgeLimitMillis are specified, then both
     * limits must be exceeded before a task can fail permanently.
     */
    private final long taskAgeLimitMillis;

    /**
     * Controls the delay before the first retry. Subsequent retries will use this
     * value adjusted according to the {@link RetryOptions#retryDelayMultiplier}.
     */
    private final long initialDelayMillis;

    /**
     * Controls the change in retry delay. The retry delay of the previous call
     * is multiplied by the multiplier to calculate the retry delay for the next call.
     */
    private final double retryDelayMultiplier;

    /**
     * MaxRetryDelay puts a limit on the value of the retry delay, so that the
     * {@link RetryOptions#retryDelayMultiplier} can't increase the retry delay
     * higher than this amount.
     */
    private final long maxRetryDelayMillis;

    private RetryOptions(Builder builder) {
        this.taskRetryLimit = builder.taskRetryLimit;
        this.taskAgeLimitMillis = builder.taskAgeLimitMillis;
        this.initialDelayMillis = builder.initialDelayMillis;
        this.retryDelayMultiplier = builder.retryDelayMultiplier;
        this.maxRetryDelayMillis = builder.maxRetryDelayMillis;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static RetryOptions empty() {
        return EMPTY;
    }

    public int getTaskRetryLimit() {
        return taskRetryLimit;
    }

    public long getTaskAgeLimitMillis() {
        return taskAgeLimitMillis;
    }

    public long getInitialDelayMillis() {
        return initialDelayMillis;
    }

    public double getRetryDelayMultiplier() {
        return retryDelayMultiplier;
    }

    public long getMaxRetryDelayMillis() {
        return maxRetryDelayMillis;
    }

    @Override
    public String toString() {
        return MoreObjects.toStringHelper(this)
                .add("taskRetryLimit", taskRetryLimit)
                .add("taskAgeLimitMillis", taskAgeLimitMillis)
                .add("initialDelayMillis", initialDelayMillis)
                .add("retryDelayMultiplier", retryDelayMultiplier)
                .add("maxRetryDelayMillis", maxRetryDelayMillis)
                .toString();
    }

    @NotThreadSafe
    public static class Builder {
        private int taskRetryLimit = 0;
        private long taskAgeLimitMillis = 0;
        private long initialDelayMillis = DEFAULT_INITIAL_DELAY_MILLIS;
        private double retryDelayMultiplier = DEFAULT_RETRY_DELAY_MULTIPLIER;
        private long maxRetryDelayMillis = DEFAULT_MAX_RETRY_DELAY_MILLIS;

        private Builder() {
        }

        private static int requirePositive(int num) {
            if (num < 0) {
                throw new IllegalArgumentException("Should be positive: " + num);
            }

            return num;
        }

        private static long requirePositive(long num) {
            if (num < 0) {
                throw new IllegalArgumentException("Should be positive: " + num);
            }

            return num;
        }

        private static double requirePositive(double num) {
            if (num < 0) {
                throw new IllegalArgumentException("Should be positive: " + num);
            }

            return num;
        }

        /**
         * @see RetryOptions#taskRetryLimit
         */
        public Builder setTaskRetryLimit(int taskRetryLimit) {
            this.taskRetryLimit = requirePositive(taskRetryLimit);
            if (taskAgeLimitMillis == 0) {
                taskAgeLimitMillis = Long.MAX_VALUE;
            }
            return this;
        }

        /**
         * @see RetryOptions#taskAgeLimitMillis
         */
        public Builder setTaskAgeLimit(long taskAgeLimitMillis, TimeUnit unit) {
            this.taskAgeLimitMillis = requirePositive(unit.toMillis(taskAgeLimitMillis));
            if (taskRetryLimit == 0) {
                taskRetryLimit = Integer.MAX_VALUE;
            }
            return this;
        }

        /**
         * @see RetryOptions#initialDelayMillis
         */
        public Builder setInitialDelay(long initialDelayMillis, TimeUnit unit) {
            this.initialDelayMillis = requirePositive(unit.toMillis(initialDelayMillis));
            return this;
        }

        /**
         * @see RetryOptions#retryDelayMultiplier
         */
        public Builder setRetryDelayMultiplier(double multiplier) {
            this.retryDelayMultiplier = requirePositive(multiplier);
            return this;
        }

        /**
         * @see RetryOptions#maxRetryDelayMillis
         */
        public Builder setMaxRetryDelay(long maxRetryDelayMillis, TimeUnit unit) {
            this.maxRetryDelayMillis = requirePositive(unit.toMillis(maxRetryDelayMillis));
            return this;
        }

        public RetryOptions build() {
            return new RetryOptions(this);
        }
    }
}
