package ru.yandex.solomon.alert.notification;

import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.evaluation.EvaluationStatusTemplateUtils;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.util.host.HostUtils;

import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
public class TemplateVarsFactory {
    private final String address;
    private final String cloudAddress;

    public TemplateVarsFactory(String address, String cloudAddress) {
        this.address = address;
        this.cloudAddress = cloudAddress;
    }

    private static Map<String, Object> makeTemplateParams(EvaluationStatus status) {
        if (status.getDescription().isEmpty()) {
            return ImmutableMap.of("code", status.getCode().name());
        }

        return ImmutableMap.of("code", status.getCode().name(), "details", status.getDescription());
    }

    public Map<String, Object> makeTemplateParams(List<Event> events) {
        SubAlert subAlert = (SubAlert) events.get(0).getAlert();
        Alert parent = subAlert.getParent();

        Map<String, Object> result = new HashMap<>();
        result.put("alert", parent);
        result.put("url", makeUrl(parent));
        result.put("senderHost", HostUtils.getFqdn());
        events.stream()
                .map(event -> event.getState().getSince())
                .min(Instant::compareTo)
                .ifPresent(since -> result.put("since", since));

        Map<EvaluationStatus.Code, List<Event>> groupedByStatus = events.stream()
                .collect(Collectors.groupingBy(event -> event.getState().getStatus().getCode()));

        for (Map.Entry<EvaluationStatus.Code, List<Event>> entry : groupedByStatus.entrySet()) {
            Map<String, Object> status = new HashMap<>();
            EvaluationStatus.Code code = entry.getKey();
            status.put("count", entry.getValue().size());
            status.put("events", entry.getValue().stream().map(this::makeTemplateParams).collect(toList()));
            result.put(code.name().toLowerCase(), status);

            List<Event> shortListEvents = entry.getValue().stream()
                .limit(3)
                .collect(toList());

            Map<String, String> shortLabels = new HashMap<>();
            for (Event e : shortListEvents) {
                shortLabels.put(makeUrl(e.getAlert()), format(e.getAlert(), labelsFromEvent(e)));
            }
            status.put("shortLabelList", shortLabels.entrySet()
                .stream()
                .sorted(Map.Entry.comparingByValue())
                .collect(Collectors.toList()));
        }
        return result;
    }

    public Map<String, Object> makeTemplateParams(Event event) {
        Labels labels = labelsFromEvent(event);

        return ImmutableMap.<String, Object>builder()
                .put("alert", event.getAlert())
                .put("labelsString", format(event.getAlert(), labels))
                .put("labels", labels.toMap())
                .put("since", event.getState().getSince())
                .put("evaluatedAt", event.getState().getLatestEval())
                .put("status", makeTemplateParams(event.getState().getStatus()))
                .put("annotations", event.getState().getStatus().getAnnotations())
                .put("annotationsList", event.getState().getStatus().getAnnotations()
                        .entrySet()
                        .stream()
                        .sorted(Map.Entry.comparingByKey())
                        .collect(Collectors.toList()))
                .put("serviceProviderAnnotations", event.getState().getStatus().getServiceProviderAnnotations())
                .put("serviceProviderAnnotationsList", event.getState().getStatus().getServiceProviderAnnotations()
                        .entrySet()
                        .stream()
                        .sorted(Map.Entry.comparingByKey())
                        .collect(Collectors.toList()))
                .put("url", makeUrl(event.getAlert()))
                .put(EvaluationStatusTemplateUtils.statusToTemplateKey(event.getState().getStatus().getCode()), true)
                .put("senderHost", HostUtils.getFqdn())
                .build();
    }

    private String format(Alert alert, Labels subAlertLabels) {
        if (alert.getAlertType() != AlertType.SUB_ALERT) {
            return "";
        }
        var groupByLabels = ((SubAlert) alert).getParent().getGroupByLabels();
        if (groupByLabels.isEmpty()) {
            return "";
        }
        Map<String, String> subAlertKv = subAlertLabels.toMap();
        return groupByLabels.stream().sorted()
                .map(key -> key + "=" + (subAlertKv.containsKey(key) ? '\'' + subAlertKv.get(key) + '\'' : '-'))
                .collect(Collectors.joining(", ", "{", "}"));
    }

    public Labels labelsFromEvent(Event event) {
        Labels labels;
        if (event.getAlert().getAlertType() == AlertType.SUB_ALERT) {
            SubAlert subAlert = (SubAlert) event.getAlert();
            labels = subAlert.getGroupKey();
        } else {
            labels = Labels.of();
        }
        return labels;
    }

    private String makeUrl(Alert alert) {
        if (alert.getAlertType() == AlertType.SUB_ALERT) {
            SubAlert subAlert = (SubAlert) alert;
            return address + "/admin/projects/" + alert.getProjectId() + "/alerts/" + subAlert.getParent().getId() + "/subAlerts/" + subAlert.getId();
        }

        return address + "/admin/projects/" + alert.getProjectId() + "/alerts/" + alert.getId();
    }

    public String makeMuteUrl(String projectId, String muteId) {
        return address + "/admin/projects/" + projectId + "/mutes/" + muteId;
    }

    public String makeCloudUrl(Alert alert) {
        if (alert.getAlertType() == AlertType.SUB_ALERT) {
            return "";
        }

        return cloudAddress + "/" + alert.getFolderId() + "/alert/" + alert.getId() + "/view";
    }
}
