package ru.yandex.solomon.alert.notification.channel;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class CompositeNotificationChannelFactory implements NotificationChannelFactory {
    private static final Logger logger = LoggerFactory.getLogger(CompositeNotificationChannelFactory.class);

    private final Map<NotificationType, NotificationChannelFactory> factoryByType;

    public CompositeNotificationChannelFactory(List<NotificationChannelFactory> factories) {
        this.factoryByType = Maps.newEnumMap(NotificationType.class);
        for (NotificationChannelFactory factory : factories) {
            for (var entry : factory.getSupportTypes().entrySet()) {
                NotificationType type = entry.getKey();
                int priority = entry.getValue();
                var prev = factoryByType.get(entry.getKey());
                if (prev == null || prev.getSupportTypes().get(type) < priority) {
                    factoryByType.put(type, factory);
                }
            }
        }

        for (NotificationType type : NotificationType.values()) {
            if (factoryByType.get(type) == null && type != NotificationType.UNKNOWN) {
                logger.error("Not configured notification type: " + type);
            }
        }
    }

    @Override
    public Map<NotificationType, Integer> getSupportTypes() {
        return factoryByType.keySet().stream()
                .collect(Collectors.toMap(Function.identity(), ignore -> Integer.MAX_VALUE));
    }

    @Nonnull
    @Override
    public NotificationChannel createChannel(Notification notification) {
        try {
            NotificationChannelFactory factory = factoryByType.get(notification.getType());
            if (factory == null) {
                throw new UnsupportedOperationException("Unsupported notification type: " + notification.getType());
            }

            return factory.createChannel(notification);
        } catch (Throwable e) {
            return new FailedInitNotificationChannel(notification, e);
        }
    }
}
