package ru.yandex.solomon.alert.notification.channel;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.cluster.broker.notification.DelegateNotificationChannel;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.domain.DevNullNotification;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DevNullOrDefaultsNotificationChannel extends AbstractNotificationChannel<DevNullNotification> {
    private final static Logger logger = LoggerFactory.getLogger(DevNullOrDefaultsNotificationChannel.class);
    private final Collection<DelegateNotificationChannel> defaultChannels;

    public DevNullOrDefaultsNotificationChannel(
            String id,
            String projectId,
            Collection<DelegateNotificationChannel> defaultChannels)
    {
        super(new DevNullNotification(id, projectId));
        this.defaultChannels = defaultChannels;
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.defaults();
    }

    @Nonnull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        List<CompletableFuture<NotificationStatus>> futures = new ArrayList<>();
        for (DelegateNotificationChannel defaultChannel : defaultChannels) {
            if (defaultChannel.getDelegate() instanceof DevNullOrDefaultsNotificationChannel) {
                logger.error("Something goes wrong for notification '{}' from '{}' with default channel '{}'", notification.getId(), notification.getProjectId(), defaultChannel.getId());
            } else {
                if (defaultChannel.getNotification().getDefaultForSeverity().isEmpty()) {
                    futures.add(defaultChannel.send(latestSuccessSend, event));
                }
            }
        }

        return CompletableFutures.allOf(futures)
                .thenApply(notificationStatus -> NotificationStatus.ABSENT_NOTIFICATION_CHANNEL);
    }

    @Override
    public boolean isDefault() {
        return true;
    }

    @Override
    public void close() {
    }
}
