package ru.yandex.solomon.alert.notification.channel;

import java.time.Duration;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;

import org.apache.commons.lang3.StringUtils;
import org.asynchttpclient.Response;

import ru.yandex.misc.io.http.HttpHeaderNames;
import ru.yandex.misc.io.http.HttpStatus;

/**
 * @author Vladimir Gordiychuk
 */
public class HttpHelper {

    public static NotificationStatus responseToStatus(Response response) {
        NotificationStatus status = classifyResponse(response);
        long retryDelay = getRetryDelayMillis(response);
        if (retryDelay != 0) {
            status = status.withRetryAfterMillis(retryDelay);
        }
        return status;
    }

    private static NotificationStatus classifyResponse(Response response) {
        String description = response.getResponseBody().isBlank() ? response.getStatusText() : response.getResponseBody();

        if (HttpStatus.is2xx(response.getStatusCode())) {
            return NotificationStatus.SUCCESS;
        }

        if (response.getStatusCode() == HttpStatus.SC_403_FORBIDDEN ||
            response.getStatusCode() == HttpStatus.SC_401_UNAUTHORIZED)
        {
            return NotificationStatus.PERMISSION_DENIED.withDescription(description);
        }

        if (response.getStatusCode() == HttpStatus.SC_429_TOO_MANY_REQUESTS) {
            return NotificationStatus.RESOURCE_EXHAUSTED.withDescription(description);
        }

        if (HttpStatus.is4xx(response.getStatusCode())) {
            return NotificationStatus.INVALID_REQUEST.withDescription(description);
        }

        if (HttpStatus.is5xx(response.getStatusCode())) {
            return NotificationStatus.ERROR_ABLE_TO_RETRY.withDescription(description);
        }

        return NotificationStatus.ERROR.withDescription(description);
    }

    public static long getRetryDelayMillis(Response response) {
        String retryAfter = response.getHeader(HttpHeaderNames.RETRY_AFTER);
        if (StringUtils.isEmpty(retryAfter)) {
            return 0;
        } else {
            return parseRetryAfter(retryAfter).toMillis();
        }
    }

    public static Duration parseRetryAfter(String retryAfter) {
        if (StringUtils.isNumeric(retryAfter)) {
            return Duration.ofSeconds(Long.parseLong(retryAfter));
        } else {
            ZonedDateTime availableAt = ZonedDateTime.parse(retryAfter, DateTimeFormatter.RFC_1123_DATE_TIME);
            ZonedDateTime now = ZonedDateTime.now();

            if (now.isAfter(availableAt)) {
                return Duration.ZERO;
            }

            long delay = availableAt.toEpochSecond() - now.toEpochSecond();
            return Duration.ofSeconds(delay);
        }
    }
}
