package ru.yandex.solomon.alert.notification.channel;

import javax.annotation.Nonnull;

import com.google.common.base.MoreObjects;

/**
 * @author Vladimir Gordiychuk
 */
public final class NotificationStatus {
    public static final NotificationStatus SKIP_REPEAT = Code.SKIP_REPEAT.toStatus();
    public static final NotificationStatus SKIP_BY_STATUS = Code.SKIP_BY_STATUS.toStatus();
    public static final NotificationStatus SUCCESS = Code.SUCCESS.toStatus();
    public static final NotificationStatus ERROR = Code.ERROR.toStatus();
    public static final NotificationStatus INVALID_REQUEST = Code.INVALID_REQUEST.toStatus();
    public static final NotificationStatus OBSOLETE = Code.OBSOLETE.toStatus();
    public static final NotificationStatus ABSENT_NOTIFICATION_CHANNEL = Code.ABSENT_NOTIFICATION_CHANNEL.toStatus();
    public static final NotificationStatus ERROR_ABLE_TO_RETRY = Code.ERROR_ABLE_TO_RETRY.toStatus();
    public static final NotificationStatus RESOURCE_EXHAUSTED = Code.RESOURCE_EXHAUSTED.toStatus();
    public static final NotificationStatus PERMISSION_DENIED = Code.PERMISSION_DENIED.toStatus();
    public static final NotificationStatus NOT_SUBSCRIBED = Code.NOT_SUBSCRIBED.toStatus();
    public static final NotificationStatus MUTED = Code.MUTED.toStatus();

    @Nonnull
    private final Code code;
    @Nonnull
    private final String description;

    /**
     * Hint on cases when external system unavailable after which millis able retry request again.
     */
    private final long retryAfterMillisHint;

    private NotificationStatus(@Nonnull Code code, @Nonnull String description, long retryAfterMillisHint) {
        this.code = code;
        this.description = description;
        this.retryAfterMillisHint = retryAfterMillisHint;
    }

    public static String formatMessage(NotificationStatus status) {
        if ("".equals(status.getDescription())) {
            return status.code.name();
        } else {
            return status.code + ": " + status.description;
        }
    }

    @Nonnull
    public Code getCode() {
        return code;
    }

    /**
     * Details about status code
     */
    @Nonnull
    public String getDescription() {
        return description;
    }

    public long getRetryAfterMillisHint() {
        return retryAfterMillisHint;
    }

    public NotificationStatus withDescription(String description) {
        return new NotificationStatus(code, description, retryAfterMillisHint);
    }

    public NotificationStatus withRetryAfterMillis(long retryAfterMillisHint) {
        return new NotificationStatus(code, description, retryAfterMillisHint);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        NotificationStatus that = (NotificationStatus) o;

        return code == that.code;
    }

    @Override
    public int hashCode() {
        return code.hashCode();
    }

    @Override
    @SuppressWarnings("CheckReturnValue")
    public String toString() {
        MoreObjects.ToStringHelper helper = MoreObjects.toStringHelper(this)
                .add("code", code);

        if (!"".equals(description)) {
            helper.add("description", description);
        }
        if (retryAfterMillisHint != 0) {
            helper.add("retryAfterMillisHint", retryAfterMillisHint);
        }
        return helper.toString();
    }

    /**
     * List of available statue for notification.
     */
    public enum Code {
        /**
         * Send notification was skipped because already notified
         */
        SKIP_REPEAT(1),

        /**
         * Skip notification because status absent in subscribe list
         */
        SKIP_BY_STATUS(2),

        /**
         * Notification successfully delivered
         */
        SUCCESS(3),

        /**
         * Not classified kind of error occurs
         */
        ERROR(4),

        /**
         * Target service reject notification because it not valid or notification channel configured
         * not correctly
         */
        INVALID_REQUEST(5),

        /**
         * Notification not valid any more
         */
        OBSOLETE(6),

        ABSENT_NOTIFICATION_CHANNEL(7),

        ERROR_ABLE_TO_RETRY(8),

        /**
         * One of the quote exhausted
         */
        RESOURCE_EXHAUSTED(9),

        PERMISSION_DENIED(10),

        NOT_SUBSCRIBED(11),

        MUTED(12)
        ;

        private final int number;

        Code(int number) {
            this.number = number;
        }

        public static Code forNumber(int num) {
            for (Code code : values()) {
                if (code.getNumber() == num) {
                    return code;
                }
            }

            throw new IllegalArgumentException("Not found code with number: " + num);
        }

        public int getNumber() {
            return number;
        }

        public NotificationStatus toStatus() {
            return new NotificationStatus(this, "", 0);
        }

        public NotificationStatus toStatus(String description) {
            return new NotificationStatus(this, description, 0);
        }
    }
}
