package ru.yandex.solomon.alert.notification.channel;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;

import javax.annotation.Nullable;

import ru.yandex.solomon.alert.domain.AlertType;
import ru.yandex.solomon.alert.domain.SubAlert;

/**
 * @author Vladimir Gordiychuk
 */
public class SubAlertEventBatchCollector implements EventBatchCollector {
    private final ConcurrentMap<String, Batch> batchByParentId = new ConcurrentHashMap<>();
    private final ScheduledExecutorService executorService;
    private final BiConsumer<List<Event>, CompletableFuture<NotificationStatus>> consumer;
    private final long flushIntervalMillis;
    private volatile boolean closed;

    public SubAlertEventBatchCollector(
            ScheduledExecutorService executorService,
            BiConsumer</* @NonEmpty */ List<Event>, CompletableFuture<NotificationStatus>> consumer)
    {
        this(executorService, consumer, Duration.ofMinutes(1));
    }

    public SubAlertEventBatchCollector(
            ScheduledExecutorService executorService,
            BiConsumer</* @NonEmpty */ List<Event>, CompletableFuture<NotificationStatus>> consumer,
            Duration flushInterval)
    {
        this.executorService = executorService;
        this.consumer = consumer;
        this.flushIntervalMillis = flushInterval.toMillis();
    }

    @Override
    public boolean supportBatch(Event event) {
        return event.getAlert().getAlertType() == AlertType.SUB_ALERT;
    }

    @Override
    public CompletableFuture<NotificationStatus> add(Event event) {
        if (closed) {
            return CompletableFuture.completedFuture(NotificationStatus.OBSOLETE.withDescription("Channel is closed"));
        }

        SubAlert subAlert = (SubAlert) event.getAlert();
        String parentId = subAlert.getParent().getId();
        Batch batch = batchByParentId.compute(parentId, (id, prev) -> {
            if (prev == null) {
                return new Batch(event);
            }

            prev.events.add(event);
            return prev;
        });

        if (batch.schedule()) {
            batch.scheduledFlush = executorService.schedule(() -> flushBatch(parentId), flushIntervalMillis, TimeUnit.MILLISECONDS);
        }

        return batch.future;
    }

    @Override
    public void close() {
        closed = true;
        Iterator<Batch> it = batchByParentId.values().iterator();
        while (it.hasNext()) {
            Batch batch = it.next();
            it.remove();
            batch.cancel();
        }
    }

    private void flushBatch(String parentId) {
        Batch batch = batchByParentId.remove(parentId);
        if (batch == null) {
            return;
        }

        if (closed) {
            batch.cancel();
            return;
        }

        try {
            consumer.accept(batch.events, batch.future);
        } catch (Throwable e) {
            batch.future.completeExceptionally(e);
        }
    }

    private static class Batch {
        /* @NonEmpty */
        private final List<Event> events = new ArrayList<>();
        private final CompletableFuture<NotificationStatus> future = new CompletableFuture<>();
        private final AtomicBoolean scheduled = new AtomicBoolean();
        @Nullable
        private volatile ScheduledFuture<?> scheduledFlush;

        Batch(Event event) {
            this.events.add(event);
        }

        public boolean schedule() {
            return scheduled.compareAndSet(false, true);
        }

        private void cancel() {
            future.complete(NotificationStatus.OBSOLETE.withDescription("Channel is closed"));
            ScheduledFuture<?> copy = scheduledFlush;
            if (copy != null) {
                copy.cancel(false);
            }
        }
    }


}
