package ru.yandex.solomon.alert.notification.channel.cloud;

import java.time.Clock;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import yandex.cloud.auth.api.AsyncCloudAuthClient;
import yandex.cloud.auth.api.Resource;
import yandex.cloud.auth.api.Subject;
import yandex.cloud.auth.api.exception.CloudAuthBadRequestException;
import yandex.cloud.auth.api.exception.CloudAuthPermissionDeniedException;
import yandex.cloud.auth.api.exception.CloudAuthUnauthenticatedException;

import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.metrics.client.cache.TimeExpiredItem;

/**
 * @author Vladimir Gordiychuk
 */
public class CloudAuthClient {
    private static final Logger logger = LoggerFactory.getLogger(CloudAuthClient.class);
    private final long EXPIRATION_MILLIS = TimeUnit.MINUTES.toMillis(10);
    private final AsyncCloudAuthClient cloudAuthClient;
    private final Clock clock;

    private static class CacheKey {
        final String folderId;
        final String userId;

        public CacheKey(String folderId, String userId) {
            this.folderId = folderId;
            this.userId = userId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            CacheKey cacheKey = (CacheKey) o;
            return folderId.equals(cacheKey.folderId) &&
                userId.equals(cacheKey.userId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(folderId, userId);
        }

        @Override
        public String toString() {
            return "CacheKey{" +
                "folderId='" + folderId + '\'' +
                ", userId='" + userId + '\'' +
                '}';
        }
    }

    private final Cache<CacheKey, TimeExpiredItem<Boolean>> usersCache;

    public CloudAuthClient(AsyncCloudAuthClient cloudAuthClient) {
        this(cloudAuthClient, Clock.systemUTC());
    }

    public CloudAuthClient(AsyncCloudAuthClient cloudAuthClient, Clock clock) {
        this.cloudAuthClient = cloudAuthClient;
        this.usersCache = CacheBuilder.newBuilder()
            .expireAfterAccess(1, TimeUnit.DAYS)
            .build();
        this.clock = clock;
    }

    public CompletableFuture<Boolean> isAuthorizedToReadData(String folderId, String userId) {
        var item = usersCache.getIfPresent(new CacheKey(folderId, userId));
        if (item == null) {
            return checkIfAuthorizedToReadData(folderId, userId);
        }

        if (!item.isExpired(now())) {
            return CompletableFuture.completedFuture(item.getPayload());
        }

        return checkIfAuthorizedToReadData(folderId, userId)
            // still notify users even if identity not working
            .completeOnTimeout(item.getPayload(), 10, TimeUnit.SECONDS)
            .exceptionally(e -> {
                logger.error("Failed to check read permission for user {} on folder {}", userId, folderId, e);
                return item.getPayload();
            });
    }

    private CompletableFuture<Boolean> checkIfAuthorizedToReadData(String folderId, String userId) {
        return cloudAuthClient.authorize(
                Subject.UserAccount.id(userId),
                Permission.DATA_READ.getSlug(),
                Resource.folder(folderId))
            .handle((subject, throwable) -> {
                boolean authorized;
                if (throwable == null) {
                    authorized = true;
                } else {
                    if (throwable instanceof CloudAuthBadRequestException) {
                        logger.warn("Querying permissions for user {} on folder {} caused bad request", userId, folderId, throwable);
                        authorized = false;
                    } else if (throwable instanceof CloudAuthUnauthenticatedException) {
                        logger.warn("Unknown user {}", userId, throwable);
                        authorized = false;
                    } else if (throwable instanceof CloudAuthPermissionDeniedException) {
                        logger.warn("Querying permissions for user {} on folder {} caused permission denied", userId, folderId, throwable);
                        authorized = false;
                    } else {
                        // Everything else signals problems with access service and should not be cached
                        // Also throwing here allows using previous cached value
                        throw new RuntimeException("Access service is not working", throwable);
                    }
                }

                usersCache.put(new CacheKey(folderId, userId), new TimeExpiredItem<>(authorized, EXPIRATION_MILLIS, now()));
                return authorized;
            });
    }

    private long now() {
        return clock.millis();
    }

}
