package ru.yandex.solomon.alert.notification.channel.cloud;

import java.io.IOException;
import java.time.Duration;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.grpc.Status;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.cloud.auth.Headers;
import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.grpc.utils.StatusRuntimeExceptionNoStackTrace;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.notification.channel.HttpHelper;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.channel.cloud.dto.NotifyDto;
import ru.yandex.solomon.alert.notification.channel.cloud.dto.NotifyDtoV1;
import ru.yandex.solomon.selfmon.failsafe.CircuitBreaker;
import ru.yandex.solomon.selfmon.failsafe.ExpMovingAverageCircuitBreaker;
import ru.yandex.solomon.selfmon.http.HttpClientMetrics;
import ru.yandex.solomon.util.actors.PingActorRunner;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class NotifyClientImpl implements NotifyClient, AutoCloseable {
    private static final int NOTIFY_209 = 209;
    private static final int NOTIFY_SCREENSHOT_FAILED_210 = 210;

    private final Logger logger = LoggerFactory.getLogger(NotifyClientImpl.class);
    private final String host;
    private final AsyncHttpClient http;
    private final TokenProvider tokenProvider;
    private final ObjectMapper mapper;
    private final HttpClientMetrics metrics;
    private final CircuitBreaker circuitBreaker = new ExpMovingAverageCircuitBreaker(0.4, TimeUnit.MILLISECONDS.toMillis(30));
    private final PingActorRunner pingActor;

    public NotifyClientImpl(String host, AsyncHttpClient http, TokenProvider tokenProvider, MetricRegistry registry, ScheduledExecutorService timer) {
        this.host = host;
        this.http = http;
        this.tokenProvider = tokenProvider;
        this.mapper = new ObjectMapper();
        this.metrics = new HttpClientMetrics("notify", registry);
        this.pingActor = PingActorRunner.newBuilder()
                .timer(timer)
                .operation("notify_ping")
                .pingInterval(Duration.ofSeconds(10))
                .backoffMaxDelay(Duration.ofMinutes(1))
                .onPing(this::ping)
                .build();
        pingActor.schedule();
    }

    private RequestBuilder newRequest(String method, String url) {
        return newRequest(method, url, UUID.randomUUID().toString());
    }

    private RequestBuilder newRequest(String method, String url, String requestId) {
        return new RequestBuilder(method)
                .setUrl(url)
                .setHeader("X-Request-Id", requestId)
                .setHeader(Headers.TOKEN_HEADER, tokenProvider.getToken())
                .setRequestTimeout(30_000);
    }

    private CompletableFuture<?> ping(int attempt) {
        if (!circuitBreaker.attemptExecution()) {
            return CompletableFuture.failedFuture(circuitBreakerException());
        }

        var endpoint = metrics.endpoint("/ping");
        var request = newRequest("GET", host + "/ping").build();

        var future = http.executeRequest(request)
            .toCompletableFuture()
            .whenComplete((response, e) -> {
                if (e != null) {
                    circuitBreaker.markFailure();
                    return;
                }

                updateCircuitBreaker(response);
                endpoint.incStatus(response.getStatusCode());

                if (HttpStatus.is2xx(response.getStatusCode())) {
                    return;
                }

                throw not200Exception(response);
            });
        endpoint.callMetrics.forFuture(future);
        return future;
    }

    private <Body> CompletableFuture<NotificationStatus> sendVia(String endpointPath, Body message) {
        try {
            String requestId = UUID.randomUUID().toString();
            var endpoint = metrics.endpoint(endpointPath);
            var request = newRequest("POST", host + endpointPath, requestId)
                .setHeader("Content-Type", "application/json")
                .setBody(mapper.writeValueAsString(message))
                .build();

            if (!circuitBreaker.attemptExecution()) {
                return CompletableFuture.completedFuture(NotificationStatus.ERROR_ABLE_TO_RETRY
                    .withRetryAfterMillis(TimeUnit.SECONDS.toMillis(30))
                    .withDescription("Circuit breaker open: " + circuitBreaker.getSummary()));
            }

            logger.info("Send to user request {}", message);
            var future = http.executeRequest(request)
                .toCompletableFuture()
                .thenApply(response -> {
                    updateCircuitBreaker(response);
                    logger.info("{} {} {}({}) status, response {}", requestId, endpointPath,
                        response.getStatusCode(), response.getStatusText(), response.getResponseBody());
                    endpoint.incStatus(response.getStatusCode());
                    if (response.getStatusCode() == HttpStatus.SC_200_OK ||
                        response.getStatusCode() == NOTIFY_SCREENSHOT_FAILED_210)
                    {
                        return NotificationStatus.SUCCESS.withDescription(response.getResponseBody());
                    }
                    if (response.getStatusCode() == NOTIFY_209) {
                        return decodeNotify209(response);
                    }
                    var status = HttpHelper.responseToStatus(response);
                    return status.withDescription(response.getResponseBody());
                })
                .whenComplete((status, e) -> {
                    if (e != null) {
                        circuitBreaker.markFailure();
                        logger.warn(endpoint + " failed", e);
                    }
                });
            endpoint.callMetrics.forFuture(future);
            return future;
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private NotificationStatus decodeNotify209(Response response) {
        var statusCode = NotificationStatus.Code.INVALID_REQUEST;

        String code = "";
        try {
            var tree = mapper.readTree(response.getResponseBody());
            // SMS and Email return status in different fields
            @Nullable var codeField = tree.get("code");
            @Nullable var statusField = tree.get("status");
            if (codeField != null) {
                code = codeField.asText();
            } else if (statusField != null) {
                code = statusField.asText();
            }
        } catch (IOException ignored) {
        }

        if ("SMS_EMPTY_PHONE".equals(code) ||
            "USER_UNSUBSCRIBED_FROM_SMS_TYPE".equals(code) ||
            "USER_UNSUBSCRIBED_FROM_MAILTYPE".equals(code))
        {
            statusCode = NotificationStatus.Code.NOT_SUBSCRIBED;
        } else if ("LIMITEXCEEDED".equals(code)) {
            statusCode = NotificationStatus.Code.RESOURCE_EXHAUSTED;
        }

        var status = statusCode.toStatus().withDescription(response.getResponseBody());
        long retryAfterMillis = HttpHelper.getRetryDelayMillis(response);
        if (retryAfterMillis != 0) {
            return status.withRetryAfterMillis(retryAfterMillis);
        }

        return status;
    }

    @Override
    public <T> CompletableFuture<NotificationStatus> sendEmail(NotifyDto<T> email) {
        return sendVia("/api/send", email);
    }

    @Override
    public <T> CompletableFuture<NotificationStatus> sendSms(NotifyDtoV1<T> sms) {
        return sendVia("/v1/send-sync", sms);
    }

    @Override
    public <T> CompletableFuture<NotificationStatus> sendPush(NotifyDtoV1<T> push) {
        return sendVia("/v1/send-sync", push);
    }

    private void updateCircuitBreaker(Response response) {
        if (HttpStatus.is5xx(response.getStatusCode())) {
            circuitBreaker.markFailure();
        } else {
            circuitBreaker.markSuccess();
        }
    }

    private RuntimeException circuitBreakerException() {
        return new StatusRuntimeExceptionNoStackTrace(Status.UNAVAILABLE.withDescription("CircuitBreaker#OPEN " + host));
    }

    private RuntimeException not200Exception(Response response) {
        String message = String.format("/ping %s(%s) status, response %s",
                response.getStatusCode(), response.getStatusText(), response.getResponseBody());
        return new StatusRuntimeExceptionNoStackTrace(Status.UNAVAILABLE.withDescription(message));

    }

    @Override
    public void close() {
        pingActor.close();
    }
}
