package ru.yandex.solomon.alert.notification.channel.cloud.push;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.google.common.annotations.VisibleForTesting;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.channel.AbstractNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.channel.cloud.AbstractCloudNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.cloud.CloudAuthClient;
import ru.yandex.solomon.alert.notification.channel.cloud.NotifyClient;
import ru.yandex.solomon.alert.notification.channel.cloud.dto.NotifyDtoV1;
import ru.yandex.solomon.alert.notification.domain.push.CloudPushNotification;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class CloudPushNotificationChannel extends AbstractCloudNotificationChannel<CloudPushNotification> {

    public CloudPushNotificationChannel(
            CloudPushNotification notification,
            @WillNotClose NotifyClient client,
            CloudAuthClient authClient)
    {
        super(notification, client, authClient);
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.statusFiltering(
            config.getNotifyAboutStatusesOrDefault(notification.getNotifyAboutStatus()),
            config.getRepeatNotificationDelayOrDefault(notification.getRepeatNotifyDelay()));
    }

    private CompletableFuture<NotificationStatus> sendUser(Event event, String iamUserId) {
        return isAuthorizedToReadData(notification.getFolderId(), iamUserId)
            .thenCompose(isAllowed -> {
                if (isAllowed) {
                    var push = Payload.create();
                    push.iamUserId = iamUserId;
                    push.data.alertStatus = event.getState().getStatus().getCode().name();
                    push.data.alertName = event.getAlert().getName();
                    push.data.alertId = event.getAlert().getId();
                    push.data.folderId = event.getAlert().getFolderId();
                    return getNotifyClient().sendPush(push);
                } else {
                    return completedFuture(NotificationStatus.PERMISSION_DENIED.withDescription(
                        "User " + iamUserId + " is not authorized to read data from " + notification.getFolderId()));
                }
            })
            .exceptionally(throwable -> NotificationStatus.ERROR.withDescription(throwable.getMessage()));
    }

    @Nonnull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        return notification.getRecipients().stream()
                .map(iamUserId -> sendUser(event, iamUserId))
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .thenApply(AbstractNotificationChannel::mergeStatuses);
    }

    @Override
    public void close() {
    }

    @VisibleForTesting
    @JsonInclude(Include.NON_NULL)
    public static class Payload {
        public String alertStatus;
        public String alertName;
        public String alertId;
        public String folderId;

        public static NotifyDtoV1<Payload> create() {
            var dto = new NotifyDtoV1<>("monitoring.alert.trigger", new Payload());
            dto.transports = List.of("web");
            return dto;
        }

        @Override
        public String toString() {
            return "Payload{" +
                    "alertStatus='" + alertStatus + '\'' +
                    ", alertName='" + alertName + '\'' +
                    ", alertId='" + alertId + '\'' +
                    ", folderId='" + folderId + '\'' +
                    '}';
        }
    }

}
