package ru.yandex.solomon.alert.notification.channel.cloud.sms;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.TemplateVarsFactory;
import ru.yandex.solomon.alert.notification.channel.AbstractNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.channel.cloud.AbstractCloudNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.cloud.CloudAuthClient;
import ru.yandex.solomon.alert.notification.channel.cloud.NotifyClient;
import ru.yandex.solomon.alert.notification.channel.cloud.dto.NotifyDtoV1;
import ru.yandex.solomon.alert.notification.domain.sms.CloudSmsNotification;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class CloudSmsNotificationChannel extends AbstractCloudNotificationChannel<CloudSmsNotification> {
    private final Logger logger = LoggerFactory.getLogger(CloudSmsNotificationChannel.class);
    private final UrlShortener urlShortener;
    private final TemplateVarsFactory templateVarsFactory;

    public CloudSmsNotificationChannel(
            CloudSmsNotification notification,
            @WillNotClose NotifyClient client,
            CloudAuthClient authClient,
            UrlShortener urlShortener,
            TemplateVarsFactory templateVarsFactory)
    {
        super(notification, client, authClient);
        this.urlShortener = urlShortener;
        this.templateVarsFactory = templateVarsFactory;
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.statusFiltering(
            config.getNotifyAboutStatusesOrDefault(notification.getNotifyAboutStatus()),
            config.getRepeatNotificationDelayOrDefault(notification.getRepeatNotifyDelay()));
    }

    private CompletableFuture<NotificationStatus> sendUser(Event event, String shortUrl, String iamUserId) {
        return isAuthorizedToReadData(notification.getFolderId(), iamUserId)
            .thenCompose(isAllowed -> {
                if (isAllowed) {
                    var sms = Payload.create();
                    sms.iamUserId = iamUserId;
                    sms.data.alertStatus = event.getState().getStatus().getCode().name();
                    sms.data.alertName = event.getAlert().getName();
                    sms.data.url = shortUrl;
                    return getNotifyClient().sendSms(sms);
                } else {
                    return completedFuture(NotificationStatus.PERMISSION_DENIED.withDescription(
                        "User " + iamUserId + " is not authorized to read data from " + notification.getFolderId()));
                }
            })
            .exceptionally(throwable -> NotificationStatus.ERROR.withDescription(throwable.getMessage()));
    }

    @Nonnull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        return makeShortUrlSafe(event)
            .thenCompose(shortUrl -> notification.getRecipients().stream()
                .map(iamUserId -> sendUser(event, shortUrl, iamUserId))
                .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                .thenApply(AbstractNotificationChannel::mergeStatuses));
    }

    private CompletableFuture<String> makeShortUrlSafe(Event event) {
        @Nullable String longUrl = templateVarsFactory.makeCloudUrl(event.getAlert());

        if (longUrl == null) {
            logger.error("cloudUrl is not set for event " + event);
            return completedFuture(null);
        }

        return urlShortener.shorten(longUrl)
            .exceptionally(e -> {
                logger.error("Failed to shorten url " + longUrl, e);
                return null;
            })
            .thenApply(url -> (url == null) ? longUrl : url);
    }

    @Override
    public void close() {
    }

    @VisibleForTesting
    @JsonInclude(Include.NON_NULL)
    public static class Payload {
        public String alertStatus;
        public String alertName;
        public String url;

        public static NotifyDtoV1<Payload> create() {
            var dto = new NotifyDtoV1<>("monitoring.alert.trigger", new Payload());
            dto.transports = List.of("sms");
            return dto;
        }

        @Override
        public String toString() {
            return "Payload{" +
                "alertStatus='" + alertStatus + '\'' +
                ", alertName='" + alertName + '\'' +
                ", url='" + url + '\'' +
                '}';
        }
    }

}
