package ru.yandex.solomon.alert.notification.channel.cloud.sms;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.failsafe.CircuitBreaker;
import ru.yandex.solomon.selfmon.failsafe.ExpMovingAverageCircuitBreaker;
import ru.yandex.solomon.selfmon.http.HttpClientMetrics;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class YaCCUrlShortener implements UrlShortener, AutoCloseable {

    private static final int MAX_LENGTH_FOR_SHORTEN_URL = 50; // Filter garbage https://st.yandex-team.ru/SOLOMON-8051
    private final Logger logger = LoggerFactory.getLogger(YaCCUrlShortener.class);
    private final String host;
    private final int requestTimeoutMillis;
    private final AsyncHttpClient http;
    private final HttpClientMetrics metrics;
    private final CircuitBreaker circuitBreaker = new ExpMovingAverageCircuitBreaker(0.4, TimeUnit.MILLISECONDS.toMillis(30));
    private final ScheduledFuture<?> periodicallyPing;

    public YaCCUrlShortener(String host, Duration requestTimeout, AsyncHttpClient http, MetricRegistry registry, ScheduledExecutorService timer) {
        this.host = host;
        this.http = http;
        this.requestTimeoutMillis = Math.toIntExact(requestTimeout.toMillis());
        this.metrics = new HttpClientMetrics("api.ya.cc", registry);
        long initialDelay = ThreadLocalRandom.current().nextLong(TimeUnit.MINUTES.toMillis(1L));
        this.periodicallyPing = timer.scheduleAtFixedRate(this::ping, initialDelay, 10_000, TimeUnit.MILLISECONDS);
    }

    private void ping() {
        if (!circuitBreaker.attemptExecution()) {
            return;
        }

        try {
            var endpoint = metrics.endpoint("/-ping-");
            var request = new RequestBuilder("GET")
                .setUrl(host + "/-ping-")
                .setHeader("X-Request-Id", UUID.randomUUID().toString())
                .setRequestTimeout(30_000)
                .build();

            var future = http.executeRequest(request)
                .toCompletableFuture()
                .whenComplete(this::onCompleteRequest)
                .whenComplete((response, e) -> {
                    if (e != null) {
                        logger.warn("/-ping- failed", e);
                        return;
                    }

                    endpoint.incStatus(response.getStatusCode());
                    if (HttpStatus.is2xx(response.getStatusCode())) {
                        return;
                    }

                    logger.info("/-ping- {}({}) status, response {}", response.getStatusCode(), response.getStatusText(), response.getResponseBody());
                });
            endpoint.callMetrics.forFuture(future);
        } catch (Throwable e) {
            circuitBreaker.markFailure();
            logger.warn("/-ping- failed", e);
        }
    }

    private static boolean isGoodShortUrl(String url) {
        return url.length() <= MAX_LENGTH_FOR_SHORTEN_URL && url.startsWith("https://");
    }

    @Override
    public CompletableFuture<String> shorten(String longUrl) {
        if (!circuitBreaker.attemptExecution()) {
            return failedFuture(new RuntimeException("CircuitBreaker is open for " + host));
        }

        final String endpointPath = "/--";
        try {
            String requestId = UUID.randomUUID().toString();
            var endpoint = metrics.endpoint(endpointPath);
            var request = new RequestBuilder("POST")
                .setUrl(host + endpointPath)
                .setHeader("Content-Type", "application/x-www-form-urlencoded")
                .setHeader("X-Request-Id", requestId)
                .setBody("url=" + URLEncoder.encode(longUrl, StandardCharsets.UTF_8))
                .setRequestTimeout(requestTimeoutMillis)
                .build();

            var future = http.executeRequest(request)
                .toCompletableFuture()
                .whenComplete(this::onCompleteRequest)
                .thenCompose(response -> {
                    String body = response.getResponseBody();
                    logger.info("{} {} {}({}) status, response {}", requestId, endpointPath,
                        response.getStatusCode(), response.getStatusText(), body);
                    endpoint.incStatus(response.getStatusCode());
                    if (response.getStatusCode() == 200) {
                        if (isGoodShortUrl(body)) {
                            return completedFuture(body);
                        }
                        logger.error("{} failed to shorten url {} and replied with garbage", host, longUrl);
                        return failedFuture(new RuntimeException("garbage from " + host));
                    }
                    return failedFuture(new RuntimeException(host + " failed to shorten url"));
                });
            endpoint.callMetrics.forFuture(future);
            return future;
        } catch (Throwable e) {
            circuitBreaker.markFailure();
            return failedFuture(e);
        }
    }

    private void onCompleteRequest(Response response, @Nullable Throwable e) {
        if (e != null) {
            circuitBreaker.markFailure();
            return;
        }

        try {
            if (HttpStatus.is5xx(response.getStatusCode())) {
                circuitBreaker.markFailure();
            } else {
                circuitBreaker.markSuccess();
            }
        } catch (Throwable e2) {
            circuitBreaker.markFailure();
            throw new RuntimeException(e2);
        }
    }

    @Override
    public void close() {
        periodicallyPing.cancel(false);
    }
}
