package ru.yandex.solomon.alert.notification.channel.datalens;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonInclude.Include;
import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.domain.threshold.PredicateRule;
import ru.yandex.solomon.alert.domain.threshold.ThresholdAlert;
import ru.yandex.solomon.alert.graph.GraphLoader;
import ru.yandex.solomon.alert.graph.Line;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.channel.AbstractNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.channel.cloud.NotifyClient;
import ru.yandex.solomon.alert.notification.channel.cloud.dto.NotifyDto;
import ru.yandex.solomon.alert.notification.channel.cloud.dto.Range;
import ru.yandex.solomon.alert.notification.channel.cloud.dto.ThresholdAlertGraph;
import ru.yandex.solomon.alert.notification.domain.email.DatalensEmailNotification;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.alert.rule.SimulationResult;
import ru.yandex.solomon.alert.rule.threshold.MultiplePredicateChecker;
import ru.yandex.solomon.alert.rule.threshold.ThresholdAlertSimulator;
import ru.yandex.solomon.alert.rule.threshold.WindowCheckFunctionFactory;
import ru.yandex.solomon.util.collection.Nullables;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class DatalensEmailNotificationChannel extends AbstractNotificationChannel<DatalensEmailNotification> {
    private static final Duration GRAPHS_LOAD_TIMEOUT = Duration.ofSeconds(15);
    private final Logger logger = LoggerFactory.getLogger(DatalensEmailNotificationChannel.class);
    private final NotifyClient notifyClient;
    private final GraphLoader graphLoader;

    public DatalensEmailNotificationChannel(
            DatalensEmailNotification notification,
            @WillNotClose NotifyClient client,
            GraphLoader graphLoader)
    {
        super(notification);
        this.notifyClient = client;
        this.graphLoader = graphLoader;
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.statusFiltering(
            config.getNotifyAboutStatusesOrDefault(notification.getNotifyAboutStatus()),
            config.getRepeatNotificationDelayOrDefault(notification.getRepeatNotifyDelay()));
    }

    private CompletableFuture<NotificationStatus> sendUser(Event event, ThresholdAlertGraph alertGraph, String emailAddress) {
        var email = Payload.create();
        email.receiver = emailAddress;
        email.data.alertId = event.getAlert().getId();
        email.data.projectId = event.getAlert().getProjectId();
        email.data.alertName = event.getAlert().getName();
        email.data.alertDescription = event.getAlert().getDescription();
        email.data.alertStatus = event.getState().getStatus().getCode().name();
        email.data.evaluationDate = event.getState().getSince().toEpochMilli();
        email.data.since = event.getState().getSince().toEpochMilli();
        email.data.latestEval = event.getState().getLatestEval().toEpochMilli();
        email.data.annotations = event.getState().getStatus().getAnnotations();
        email.data.serviceProviderAnnotations = event.getState().getStatus().getServiceProviderAnnotations();
        email.data.thresholdAlertGraph = alertGraph;
        return notifyClient.sendEmail(email);
    }

    @Nonnull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        return resolveAlertGraph(event)
            .thenCompose(alertGraph -> notification.getRecipients().stream()
                    .map(emailAddress -> sendUser(event, alertGraph, emailAddress))
                    .collect(collectingAndThen(toList(), CompletableFutures::allOf))
                    .thenApply(AbstractNotificationChannel::mergeStatuses));
    }

    private CompletableFuture<ThresholdAlertGraph> resolveAlertGraph(Event event) {
        Alert alert = event.getAlert();
        EvaluationState state = event.getState();

        if (alert instanceof ThresholdAlert) {
            return resolveThresholdAlertGraph((ThresholdAlert) alert, state)
                .handle((result, e) -> {
                    if (e != null) {
                        logger.error("Exception while building graph data for cloud email", e);
                        return null;
                    }
                    return result;
                });
        }

        return completedFuture(null);
    }

    /* The code below will be completely replaced later, so copy-paste from cloud email is legal */

    private CompletableFuture<ThresholdAlertGraph> resolveThresholdAlertGraph(ThresholdAlert alert, EvaluationState state) {
        try {
            return resolveThresholdAlertGraphUnsafe(alert, state);
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    private CompletableFuture<ThresholdAlertGraph> resolveThresholdAlertGraphUnsafe(ThresholdAlert alert, EvaluationState state) {
        List<PredicateRule> predicateRules = alert.getPredicateRules();
        SimulationResult validationResult = ThresholdAlertSimulator.validatePredicateRules(predicateRules);
        if (validationResult != null) {
            logger.error("Bad alert in datalens email channel, validation error: {}", validationResult);
            return completedFuture(null);
        }

        PredicateRule alarmRule = predicateRules.get(0);
        @Nullable PredicateRule warnRule = (predicateRules.size() > 1) ? predicateRules.get(1) : null;

        Duration viewport = alert.getPeriod().plus(alert.getPeriod().dividedBy(2));
        Instant to = state.getLatestEval().minusSeconds(alert.getDelaySeconds());
        Instant from = to.minus(viewport);
        return graphLoader.load(alert, from, to, Instant.now().plus(GRAPHS_LOAD_TIMEOUT))
            .thenApply(timeSeriesList -> {
                ThresholdAlertGraph dto = new ThresholdAlertGraph();
                dto.alarmLevel = alarmRule.getThreshold();
                dto.warnLevel = Nullables.map(warnRule, PredicateRule::getThreshold);
                dto.comparison = alarmRule.getComparison().name();
                dto.aggregation = alarmRule.getThresholdType().name();
                dto.nowMillis = state.getLatestEval().toEpochMilli() - TimeUnit.SECONDS.toMillis(alert.getDelaySeconds());
                dto.windowMillis = alert.getPeriod().toMillis();
                dto.range = new Range(from.toEpochMilli(), to.toEpochMilli());
                dto.lines = new ArrayList<>();
                var checkFunctions = alert.getPredicateRules().stream()
                    .map(WindowCheckFunctionFactory::prepare)
                    .collect(toList());
                for (var timeseries : timeSeriesList) {
                    Line line = Line.of(timeseries);
                    if (line == null) {
                        throw new RuntimeException("Unsupported time series with mask " + timeseries.getSource().columnSetMask());
                    }
                    var check = MultiplePredicateChecker.checkMultipleFunctions(alert.getNoPointsPolicy(),
                            checkFunctions, timeseries.slice(dto.nowMillis - dto.windowMillis, dto.nowMillis));
                    line.status = check.getStatusCode().name();
                    line.value = check.getValue();
                    dto.lines.add(line);
                }
                return dto;
            });
    }

    @Override
    public void close() {
    }

    @VisibleForTesting
    @JsonInclude(Include.NON_NULL)
    public static class Payload {
        public String alertStatus;
        public String alertName;
        public String alertDescription;
        public String alertId;
        public String projectId;
        public long evaluationDate;
        public long since;
        public long latestEval;
        public Map<String, String> annotations;
        public Map<String, String> serviceProviderAnnotations;
        public ThresholdAlertGraph thresholdAlertGraph;

        public static NotifyDto<Payload> create() {
            return new NotifyDto<>("datalens-alert-triggered", new Payload());
        }

        @Override
        public String toString() {
            return "Payload{" +
                "alertStatus='" + alertStatus + '\'' +
                ", alertName='" + alertName + '\'' +
                ", alertDescription='" + alertDescription + '\'' +
                ", alertId='" + alertId + '\'' +
                ", projectId='" + projectId + '\'' +
                ", evaluationDate=" + evaluationDate +
                ", since=" + since +
                ", latestEval=" + latestEval +
                ", annotations=" + annotations +
                ", serviceProviderAnnotations=" + serviceProviderAnnotations +
                ", thresholdAlertGraph=" + thresholdAlertGraph +
                '}';
        }
    }

}
