package ru.yandex.solomon.alert.notification.channel.email;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;
import javax.mail.Message;

import com.google.common.base.Throwables;

import ru.yandex.commune.mail.ContentType;
import ru.yandex.commune.mail.MailMessage;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.inject.spring.notification.EmailTemplate;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.TemplateVarsFactory;
import ru.yandex.solomon.alert.notification.channel.AbstractNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.EventBatchCollector;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.channel.SubAlertEventBatchCollector;
import ru.yandex.solomon.alert.notification.domain.email.EmailNotification;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class EmailNotificationChannel extends AbstractNotificationChannel<EmailNotification> {
    @WillNotClose
    private final EmailClient emailClient;

    private final EmailTemplate template;
    private final String recipients;

    private final EventBatchCollector batchCollector;
    private final TemplateVarsFactory variablesFactory;

    public EmailNotificationChannel(
            EmailNotification notification,
            @WillNotClose EmailClient client,
            @WillNotClose ScheduledExecutorService executorService,
            TemplateVarsFactory variablesFactory,
            EmailTemplate template)
    {
        super(notification);
        this.emailClient = client;
        this.variablesFactory = variablesFactory;
        this.template = template;
        this.recipients = notification.getRecipients()
                .stream()
                .collect(Collectors.joining(", "));
        this.batchCollector = new SubAlertEventBatchCollector(executorService, this::sendMany);
    }

    private static NotificationStatus classifyError(Throwable e) {
        // TODO: classify error (gordiychuk@)
        return NotificationStatus.ERROR
                .withDescription(Throwables.getStackTraceAsString(e));
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.statusFiltering(
            config.getNotifyAboutStatusesOrDefault(notification.getNotifyAboutStatus()),
            config.getRepeatNotificationDelayOrDefault(notification.getRepeatNotifyDelay()));
    }

    @Nonnull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        if (batchCollector.supportBatch(event)) {
            return batchCollector.add(event);
        }

        MailMessage email;
        try {
            email = prepareEmail(event);
        } catch (Throwable e) {
            return CompletableFuture.completedFuture(classifyError(e));
        }

        return emailClient.send(email)
                .exceptionally(EmailNotificationChannel::classifyError);
    }

    @Override
    public void close() {
        batchCollector.close();
    }

    private void sendMany(List<Event> events, CompletableFuture<NotificationStatus> future) {
        MailMessage email;
        try {
            email = prepareEmail(events);
        } catch (Throwable e) {
            future.complete(classifyError(e));
            return;
        }

        emailClient.send(email)
                .exceptionally(EmailNotificationChannel::classifyError)
                .thenAccept(future::complete);
    }

    private MailMessage prepareEmail(Event event) {
        Map<String, Object> params = variablesFactory.makeTemplateParams(event);
        return MailMessage.empty()
                .withFrom(template.getFrom())
                .withSubject(template.getSubject(params))
                .withContent(template.getContent(params), ContentType.TEXT_HTML_UTF8)
                .addRecipient(Message.RecipientType.TO, recipients);
    }

    private MailMessage prepareEmail(List<Event> events) {
        Map<String, Object> params = variablesFactory.makeTemplateParams(events);
        return MailMessage.empty()
                .withFrom(template.getFrom())
                .withSubject(template.getMultiAlertSubject(params))
                .withContent(template.getMultiAlertContent(params), ContentType.TEXT_HTML_UTF8)
                .addRecipient(Message.RecipientType.TO, recipients);
    }
}
