package ru.yandex.solomon.alert.notification.channel.email;

import java.time.Instant;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.annotation.Nullable;
import javax.mail.MessagingException;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.MimeMessage;

import com.google.common.base.Throwables;
import com.sun.mail.smtp.SMTPSendFailedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.commune.mail.MailMessage;
import ru.yandex.commune.mail.MailUtils;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;

/**
 * @author Vladimir Gordiychuk
 */
public class JavamailTransport implements MailTransport, AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(JavamailTransport.class);

    private final ArrayBlockingQueue<Task> queue;
    private final ExecutorService executor;
    private final JavamailTransportMetrics metrics;
    private final List<Thread> senders;
    private volatile boolean closed;

    public JavamailTransport(Properties properties, int connections, int queueCapacity, ExecutorService executor, MetricRegistry registry) {
        this.queue = new ArrayBlockingQueue<>(queueCapacity);
        this.executor = executor;
        this.metrics = new JavamailTransportMetrics(registry);
        this.senders = IntStream.range(0, connections)
                .mapToObj(ignore -> {
                    MailSender sender = new MailSender(properties);
                    Thread thread = new Thread(sender, JavamailTransport.class.getSimpleName() + "-" + ignore);
                    thread.setDaemon(true);
                    thread.start();
                    return thread;
                })
                .collect(Collectors.toList());
    }

    @Override
    public CompletableFuture<NotificationStatus> send(MailRequest request) {
        if (closed) {
            return CompletableFuture.completedFuture(NotificationStatus.ERROR.withDescription("Shutdown in process"));
        }

        CompletableFuture<NotificationStatus> future = new CompletableFuture<>();
        Task task = new Task(request, future);
        metrics.incQueueSize();
        if (!queue.offer(task)) {
            metrics.queueOverflow();
            metrics.decQueueSize();
            future.complete(NotificationStatus.RESOURCE_EXHAUSTED.withDescription("Send email queue overflow"));
        }

        return future;
    }

    public JavamailTransportMetrics getMetrics() {
        return metrics;
    }

    @Override
    public void close() {
        closed = true;
        senders.forEach(Thread::interrupt);
    }

    private static class Task {
        private final MailRequest request;
        private final CompletableFuture<NotificationStatus> future;

        public Task(MailRequest request, CompletableFuture<NotificationStatus> future) {
            this.request = request;
            this.future = future;
        }
    }

    private class MailSender implements Runnable {
        private final Session session;
        @Nullable
        private Transport transport;

        public MailSender(Properties properties) {
            this.session = Session.getInstance(properties);
        }

        @Override
        public void run() {
            while (!closed) {
                try {
                    Task task = queue.take();
                    metrics.decQueueSize();
                    try {
                        processTask(task);
                    } catch (Throwable e) {
                        logger.error("Failed send email task processing: " + task, e);
                        completeExceptionally(task, e);
                    }
                } catch (InterruptedException e) {
                    // ok
                }
            }

            Transport copy = transport;
            if (copy != null) {
                try {
                    copy.close();
                } catch (Throwable e) {
                    logger.error("Failed to close mail transport", e);
                }
            }
        }

        private void processTask(Task task) {
            if (task.request.isCanceled() || closed) {
                complete(task, NotificationStatus.OBSOLETE);
                return;
            }

            long now = System.currentTimeMillis();
            if (now >= task.request.getDeadline()) {
                complete(task, NotificationStatus.OBSOLETE
                        .withDescription("Deadline exceeded at: " + Instant.ofEpochMilli(task.request.getDeadline())));
                return;
            }

            try {
                syncSend(task.request.getMessage());
                complete(task, NotificationStatus.SUCCESS);
            } catch (Throwable e) {
                logger.warn("Failed send email", e);
                NotificationStatus status = classifyError(e);
                complete(task, status);
            }
        }

        private void syncSend(MailMessage message) {
            long startTimeNanos = System.nanoTime();
            try {
                Transport transport = ensureConnected();
                MimeMessage msg = MailUtils.toMimeMessage(session, message);
                msg.saveChanges();
                transport.sendMessage(msg, msg.getAllRecipients());
            } catch (Exception e) {
                throw MailUtils.translate(e);
            } finally {
                metrics.completeSend(TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTimeNanos));
            }
        }

        private Transport ensureConnected() throws MessagingException {
            Transport copy = transport;
            if (copy == null) {
                copy = session.getTransport();
                transport = copy;
                copy.connect();
            } else if (!copy.isConnected()) {
                copy.close();
                copy = session.getTransport();
                copy.connect();
                transport = copy;
            }

            return copy;
        }

        private NotificationStatus classifyError(Throwable t) {
            Throwable cause = t;
            while (cause != null) {
                if (cause instanceof SMTPSendFailedException) {
                    SMTPSendFailedException smtp = (SMTPSendFailedException) cause;
                    metrics.failedSmtpRequest(smtp.getReturnCode());

                    int code = smtp.getReturnCode();
                    if (is4xx(code)) {
                        return NotificationStatus.ERROR_ABLE_TO_RETRY
                                .withDescription(cause.getMessage())
                                .withRetryAfterMillis(TimeUnit.MINUTES.toMillis(2));
                    } else if (is5xx(code)) {
                        return NotificationStatus.INVALID_REQUEST.withDescription(cause.getMessage());
                    } else {
                        return NotificationStatus.ERROR.withDescription(cause.getMessage());
                    }
                }
                cause = cause.getCause();
            }

            return NotificationStatus.ERROR.withDescription(Throwables.getStackTraceAsString(t));
        }

        private boolean is4xx(int code) {
            return code >= 400 && code <= 499;
        }

        private boolean is5xx(int code) {
            return code >= 500 && code <= 599;
        }

        private void complete(Task task, NotificationStatus status) {
            executor.execute(() -> task.future.complete(status));
        }

        private void completeExceptionally(Task task, Throwable e) {
            executor.execute(() -> task.future.completeExceptionally(e));
        }
    }
}
