package ru.yandex.solomon.alert.notification.channel.fallback;

import java.time.Instant;
import java.util.EnumSet;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jetbrains.annotations.NotNull;

import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationChannel;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;
import ru.yandex.solomon.flags.FeatureFlag;
import ru.yandex.solomon.flags.FeatureFlagsHolder;

@ParametersAreNonnullByDefault
public class FallbackNotificationChannel implements NotificationChannel {
    private final NotificationChannel tgChannel;
    private final NotificationChannel ymChannel;
    private final FeatureFlagsHolder flagsHolder;

    public FallbackNotificationChannel(
            NotificationChannel tgChannel,
            NotificationChannel ymChannel,
            FeatureFlagsHolder flagsHolder)
    {
        this.tgChannel = tgChannel;
        this.ymChannel = ymChannel;
        this.flagsHolder = flagsHolder;
    }

    private static class AtLeastOneSuccessCompleter {
        private static final EnumSet<NotificationStatus.Code> COMPLETING_CODES = EnumSet.of(
                NotificationStatus.Code.SUCCESS
                // There are many NotificationStatus.Codes that may be considered
                // completing, but they are utility and used to drive Notification FSM.
                // They never occur as a real NotificationChannel::send result
        );

        private final CompletableFuture<NotificationStatus> future;
        private final AtomicInteger inflightRequests;

        public AtLeastOneSuccessCompleter(int totalRequests) {
            future = new CompletableFuture<>();
            inflightRequests = new AtomicInteger(totalRequests);
        }

        public void consume(NotificationStatus notificationStatus, @Nullable Throwable throwable) {
            boolean lastResponse = inflightRequests.decrementAndGet() == 0;
            if (throwable == null && COMPLETING_CODES.contains(notificationStatus.getCode())) {
                future.complete(notificationStatus);
                return;
            }
            if (lastResponse) {
                if (throwable != null) {
                    future.completeExceptionally(throwable);
                } else {
                    future.complete(notificationStatus);
                }
            }
        }

        public CompletableFuture<NotificationStatus> getFuture() {
            return future;
        }
    }

    @NotNull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        boolean duplicate = flagsHolder.hasFlag(FeatureFlag.DUPLICATE_TELEGRAM_TO_MESSENGER, getProjectId());
        if (!duplicate) {
            return tgChannel.send(latestSuccessSend, event);
        }
        var completer = new AtLeastOneSuccessCompleter(2);
        tgChannel.send(latestSuccessSend, event).whenComplete(completer::consume);
        ymChannel.send(latestSuccessSend, event).whenComplete(completer::consume);
        return completer.getFuture();
    }

    @NotNull
    @Override
    public String getId() {
        return tgChannel.getId();
    }

    @NotNull
    @Override
    public String getProjectId() {
        return tgChannel.getProjectId();
    }

    @Override
    public NotificationType getType() {
        return tgChannel.getType();
    }

    @Override
    public DispatchRule getDispatchRule(ChannelConfig configOverride) {
        return tgChannel.getDispatchRule(configOverride);
    }

    @Override
    public void close() {
        tgChannel.close();
        ymChannel.close();
    }

    @Override
    public Notification getNotification() {
        return tgChannel.getNotification();
    }

    @Override
    public boolean isDefault() {
        return tgChannel.isDefault();
    }
}
