package ru.yandex.solomon.alert.notification.channel.juggler;

import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import ru.yandex.juggler.client.JugglerClient;
import ru.yandex.juggler.client.JugglerDirectClient;
import ru.yandex.juggler.client.JugglerDirectClientOptions;
import ru.yandex.juggler.resolver.FixedTargetsResolver;
import ru.yandex.juggler.resolver.HttpProxyResolver;
import ru.yandex.juggler.resolver.ProxyResolveObserver;
import ru.yandex.juggler.resolver.ProxyResolver;
import ru.yandex.juggler.resolver.SinglePushWrapperResolver;
import ru.yandex.juggler.target.JugglerTargetOptions;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.util.RateLimit;
import ru.yandex.solomon.alert.util.RateLimiters;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.TimeUnitConverter;
import ru.yandex.solomon.config.protobuf.Time;
import ru.yandex.solomon.config.protobuf.alert.JugglerChannelConfig;
import ru.yandex.solomon.flags.FeatureFlagsHolder;
import ru.yandex.solomon.selfmon.failsafe.CircuitBreaker;
import ru.yandex.solomon.selfmon.failsafe.ExpMovingAverageCircuitBreaker;
import ru.yandex.solomon.selfmon.failsafe.NoOpCircuitBreaker;
import ru.yandex.solomon.util.file.FileStorage;

/**
 * @author Vladimir Gordiychuk
 */
public final class JugglerClients {
    private JugglerClients() {
    }

    public static JugglerClient create(
            JugglerChannelConfig config,
            MetricRegistry registry,
            Executor httpExecutor,
            ScheduledExecutorService timer,
            FeatureFlagsHolder flagsHolder)
    {
        RateLimit rateLimit = makeRateLimiter(config);
        JugglerClient client = makeBatchClient(config, registry, httpExecutor, timer, flagsHolder);
        return new RateLimitedJugglerClient(client, rateLimit);
    }

    public static ProxyResolver makeProxyResolver(
            FileStorage storage,
            ProxyResolveObserver resolveObserver,
            JugglerChannelConfig config,
            MetricRegistry registry,
            Executor httpExecutor,
            ScheduledExecutorService timer)
    {
        switch (config.getDeliveryMethodCase()) {
            case DISCOVERY_ADDRESS_PAIR:
                return new HttpProxyResolver(
                        storage,
                        config.getDiscoveryAddressPair().getGlobalDiscoveryAddress(),
                        config.getDiscoveryAddressPair().getRegionalDiscoveryAddress(),
                        registry,
                        timer,
                        httpExecutor,
                        resolveObserver);
            case ADDRESS:
                return new SinglePushWrapperResolver(config.getAddress(), resolveObserver);
            case BACKENDDISCOVERYADDRESS:
                return new HttpProxyResolver(
                        storage,
                        "",
                        config.getBackendDiscoveryAddress(),
                        registry,
                        timer,
                        httpExecutor,
                        resolveObserver);
            case BACKENDSCONFIG:
                return new FixedTargetsResolver(config.getBackendsConfig(), resolveObserver);
        }
        throw new IllegalArgumentException("Delivery method is not configured for JugglerChannelConfig");
    }

    private static JugglerClient makeBatchClient(
            JugglerChannelConfig config,
            MetricRegistry registry,
            Executor httpExecutor,
            ScheduledExecutorService timer,
            FeatureFlagsHolder flagsHolder)
    {
        JugglerDirectClientOptions clientOptions = new JugglerDirectClientOptions()
                .setRegistry(registry)
                .setRelayCircuitBreakerFactory(() -> makeCircuitBreaker(config))
                .setRelayHttpExecutor(httpExecutor);

        JugglerTargetOptions targetOptions = new JugglerTargetOptions()
                .setApplicationName("solomon")
                .setTimer(timer)
                .setFlusherExecutor(httpExecutor);

        setMillisIfNotEmpty(config.getFlushInterval(), targetOptions::setFlushIntervalMillis);
        if (config.getSendQueueSize() != 0) {
            targetOptions.setQueueCapacity(config.getSendQueueSize());
        }

        return new JugglerDirectClient(clientOptions, targetOptions, flagsHolder);
    }

    private static void setMillisIfNotEmpty(Time time, Consumer<Long> consumer) {
        if (time == Time.getDefaultInstance()) {
            return;
        }

        TimeUnit unit = TimeUnitConverter.protoToUnit(time.getUnit());
        consumer.accept(unit.toMillis(time.getValue()));
    }

    private static RateLimit makeRateLimiter(JugglerChannelConfig config) {
        int limit = Math.toIntExact(config.getRateLimitEventsPerSecond());
        if (limit == 0) {
            return RateLimiters.noop();
        } else {
            return RateLimiters.create(limit);
        }
    }

    private static CircuitBreaker makeCircuitBreaker(JugglerChannelConfig config) {
        double threshold = config.getCircuitBreakerFailureQuantileThreshold();
        if (Double.compare(threshold, 0d) == 0) {
            return NoOpCircuitBreaker.INSTANCE;
        }

        long resetTimeout = TimeConverter.protoToDuration(config.getCircuitBreakerResetTimeout(), java.time.Duration.ofSeconds(30))
                .toMillis();

        return new ExpMovingAverageCircuitBreaker(threshold, resetTimeout);
    }
}
