package ru.yandex.solomon.alert.notification.channel.juggler;

import java.io.IOException;
import java.time.Clock;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Strings;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.juggler.client.JugglerClient;
import ru.yandex.juggler.dto.EventStatus;
import ru.yandex.juggler.dto.JugglerEvent;
import ru.yandex.juggler.dto.JugglerEvent.Status;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.domain.AlertKey;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.inject.spring.TemplateUtils;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.TemplateVarsFactory;
import ru.yandex.solomon.alert.notification.channel.AbstractNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.domain.juggler.JugglerNotification;
import ru.yandex.solomon.alert.rule.AlertProcessingState;
import ru.yandex.solomon.alert.template.Template;
import ru.yandex.solomon.alert.template.TemplateFactory;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.solomon.util.StringUtils.replaceChars;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class JugglerNotificationChannel extends AbstractNotificationChannel<JugglerNotification> {
    private static final String DEFAULT_DESCRIPTION_TEMPLATE_RESOURCE = "/ru/yandex/solomon/alert/notification/channel/template/juggler/defaultJugglerDescription.mustache";
    private static final Logger logger = LoggerFactory.getLogger(JugglerNotificationChannel.class);
    private static final long EVENT_FRESH_TIME_MILLIS = TimeUnit.MINUTES.toMillis(5L);

    private final JugglerClient client;
    private final TemplateVarsFactory variablesFactory;

    private final Clock clock;
    private final Template host;
    private final Template service;
    private final Template instance;
    private final Template description;
    private final List<Template> tags;

    public JugglerNotificationChannel(
            Clock clock,
            JugglerNotification notification,
            JugglerClient client,
            TemplateFactory templateFactory,
            TemplateVarsFactory variablesFactory)
    {
        super(notification);
        this.clock = clock;
        this.client = client;
        this.variablesFactory = variablesFactory;

        if (StringUtils.isEmpty(notification.getHost())) {
            this.host = (ignore) -> "solomon-alert";
        } else {
            this.host = templateFactory.createTemplate(notification.getHost());
        }

        if (StringUtils.isEmpty(notification.getService())) {
            this.service = templateFactory.createTemplate("{{alert.id}}");
        } else {
            this.service = templateFactory.createTemplate(notification.getService());
        }

        if (StringUtils.isEmpty(notification.getInstance())) {
            this.instance = ignore -> "";
        } else {
            this.instance = templateFactory.createTemplate(notification.getInstance());
        }

        if (StringUtils.isEmpty(notification.getJugglerDescription())) {
            try {
                this.description = TemplateUtils.makeTemplateAsResource(templateFactory, DEFAULT_DESCRIPTION_TEMPLATE_RESOURCE);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }
        } else {
            this.description = templateFactory.createTemplate(notification.getJugglerDescription());
        }

        this.tags = notification.getTags()
                .stream()
                .map(templateFactory::createTemplate)
                .collect(Collectors.toList());
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.notifyRepeating(
            config.getNotifyAboutStatusesOrDefault(notification.getNotifyAboutStatus()));
    }

    @Nonnull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        if (isFresh(event)) {
            return completedFuture(NotificationStatus.OBSOLETE
                    .withDescription("Evaluation lag " + event.getState().getLatestEval() + " greater that five minute"));
        }

        return client.sendEvent(makeJugglerEvent(event))
                .thenApply(response -> {
                    NotificationStatus status = classifyStatus(response);
                    logger.debug("{} - for {} response {}", status, notification.getId(), response);
                    return status;
                });
    }

    private boolean isFresh(Event event) {
        long eventTime = event.getState().getLatestEval().toEpochMilli();
        long lag = clock.millis() - eventTime;
        return lag > EVENT_FRESH_TIME_MILLIS;
    }

    private NotificationStatus classifyStatus(EventStatus status) {
        if (HttpStatus.is2xx(status.code)) {
            return NotificationStatus.SUCCESS;
        }

        if (HttpStatus.SC_429_TOO_MANY_REQUESTS == status.code) {
            return NotificationStatus.RESOURCE_EXHAUSTED.withDescription(status.message);
        }

        if (HttpStatus.is4xx(status.code)) {
            return NotificationStatus.INVALID_REQUEST.withDescription(status.message);
        }

        if (HttpStatus.is5xx(status.code)) {
            return NotificationStatus.ERROR_ABLE_TO_RETRY.withDescription(status.message);
        }

        return NotificationStatus.ERROR.withDescription(status.message);
    }

    private JugglerEvent makeJugglerEvent(Event event) {
        Map<String, Object> params = variablesFactory.makeTemplateParams(event);
        return new JugglerEvent(
                replaceInvalidChars(host.process(params)),
                replaceInvalidChars(service.process(params)),
                replaceInvalidChars(instance.process(params)),
                evalStatusToJugglerStatus(event.getAlertProcessingState()),
                description.process(params),
                processTags(event.getAlert().getKey(), params),
                event.getState().getLatestEval().getEpochSecond(),
                event.getAlert().getKey().getProjectId()
        );
    }

    private String replaceInvalidChars(String chars) {
        return replaceChars(chars, JugglerNotificationChannel::jugglerInvalidChar, '_');
    }

    private String replaceInvalidCharsInTag(String chars) {
        return replaceChars(chars, JugglerNotificationChannel::jugglerInvalidTagChar, '_');
    }

    private List<String> processTags(AlertKey key, Map<String, Object> params) {
        List<String> result = new ArrayList<>(3 + tags.size());
        for (var tagTemplate : tags) {
            var tagResult = tagTemplate.process(params);
            for (var tag : StringUtils.split(tagResult,',')) {
                result.add(replaceInvalidCharsInTag(tag.trim()));
            }
        }

        result.add("monitoring_project_id_" + replaceInvalidCharsInTag(key.getProjectId()));
        if (Strings.isNullOrEmpty(key.getParentId())) {
            result.add("monitoring_alert_id_" + replaceInvalidCharsInTag(key.getAlertId()));
        } else {
            result.add("monitoring_alert_id_" + replaceInvalidCharsInTag(key.getParentId()));
            result.add("monitoring_sub_alert_id_" + replaceInvalidCharsInTag(key.getAlertId()));
        }

        return result;
    }

    private Status evalStatusToJugglerStatus(AlertProcessingState processingState) {
        if (processingState.alertMuteStatus().isMuted()) {
            return Status.OK;
        }

        EvaluationStatus evalStatus = processingState.evaluationState().getStatus();
        String color = evalStatus.getAnnotations().get("trafficLight.color");
        if (color != null) {
            if ("red".equals(color)) {
                return Status.CRIT;
            }

            if ("yellow".equals(color)) {
                return Status.WARN;
            }

            if ("green".equals(color)) {
                return Status.OK;
            }
        }

        return switch (evalStatus.getCode()) {
            case OK -> Status.OK;
            case NO_DATA -> Status.INFO;
            case WARN, ERROR -> Status.WARN;
            case ALARM -> Status.CRIT;
        };
    }

    @Override
    public void close() {
    }

    @SuppressWarnings({"OverlyComplexBooleanExpression", "OverlyComplexMethod"})
    private static boolean jugglerInvalidChar(char ch) {
        return !(ch >= 'a' && ch <= 'z'
            || ch >= 'A' && ch <= 'Z'
            || ch >= '0' && ch <= '9'
            || ch == '_' || ch == '+' || ch == '-' || ch == '.' || ch == '/');
    }

    @SuppressWarnings("OverlyComplexBooleanExpression")
    private static boolean jugglerInvalidTagChar(char ch) {
        return !(ch >= 'a' && ch <= 'z'
            || ch >= 'A' && ch <= 'Z'
            || ch >= '0' && ch <= '9'
            || ch == '_' || ch == '+' || ch == '-' || ch == '.');
    }
}
