package ru.yandex.solomon.alert.notification.channel.phone;

import java.net.http.HttpResponse;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Throwables;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.jns.client.HttpJnsClient;
import ru.yandex.jns.client.JnsClient;
import ru.yandex.jns.dto.JnsSendMessageRequest;
import ru.yandex.jns.dto.JnsSendRecipient;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.solomon.alert.domain.Alert;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.domain.SubAlert;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.channel.AbstractNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.domain.PhoneNotification;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.config.protobuf.alert.JNSChannelConfig;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Alexey Trushkin
 */
@ParametersAreNonnullByDefault
public class PhoneNotificationChannel extends AbstractNotificationChannel<PhoneNotification> {
    private static final Logger logger = LoggerFactory.getLogger(PhoneNotificationChannel.class);
    private final JnsClient jnsClient;
    private final JNSChannelConfig config;

    public PhoneNotificationChannel(PhoneNotification notification, JnsClient jnsClient, JNSChannelConfig config) {
        super(notification);
        this.jnsClient = jnsClient;
        this.config = config;
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.statusFiltering(
                config.getNotifyAboutStatusesOrDefault(notification.getNotifyAboutStatus()),
                config.getRepeatNotificationDelayOrDefault(notification.getRepeatNotifyDelay()));
    }

    @Nonnull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        EvaluationState state = event.getState();
        try {
            var request = prepareRequest(event, latestSuccessSend);
            return jnsClient.sendMessage(request)
                    .thenApply(unused -> NotificationStatus.SUCCESS)
                    .exceptionally(throwable -> classifyError(throwable, request));
        } catch (Throwable e) {
            errorLogStatus(NotificationStatus.INVALID_REQUEST, state, e);
            return completedFuture(NotificationStatus.INVALID_REQUEST.withDescription(Throwables.getStackTraceAsString(e)));
        }
    }

    private JnsSendMessageRequest prepareRequest(Event event, Instant latestSuccessSend) {
        JnsSendRecipient.InternalUser user;
        if (notification.isLogin()) {
            user = JnsSendRecipient.login(notification.getLogin());
        } else {
            user = JnsSendRecipient.duty(notification.getDuty().abcService(), notification.getDuty().dutySlug());
        }
        var recipient = new JnsSendRecipient();
        recipient.phone = new JnsSendRecipient.PhoneChannelOptions();
        recipient.phone.internal = List.of(user);

        Alert alert = event.getAlert();
        var message = new JnsSendMessageRequest();
        message.projectWithTemplate = config.getProject();
        message.template = config.getTemplate();
        message.idempotencyKey = Objects.hash(
                config.getProject(), config.getTemplate(),
                alert.getProjectId(), alert.getId(),
                event.getState().getSince(), latestSuccessSend) + "";
        message.targetJnsProject = alert.getProjectId();

        List<Map<String, String>> subAlertLabels = List.of();
        if (alert instanceof SubAlert sa) {
            subAlertLabels = sa.getGroupKey().stream()
                    .map(Label::getValue)
                    .map(s -> Map.of("string_value", s))
                    .collect(Collectors.toList());
        }

        message.params = Map.of(
                "alertName", Map.of("string_value", alert.getName()),
                "alertId", Map.of("string_value", alert.getId()),
                "subAlertLabels", Map.of("list_value", subAlertLabels),
                "monitoringProject", Map.of("string_value", alert.getProjectId())
        );
        message.recipient = recipient;
        return message;
    }

    @Override
    public void close() {
    }

    private void errorLogStatus(NotificationStatus status, EvaluationState state, Throwable e) {
        logger.warn("{} - for {} by {}", status, notification.getId(), state, e);
    }

    private void logStatus(NotificationStatus status, JnsSendMessageRequest request, HttpResponse<?> response) {
        if (status.getCode() == NotificationStatus.Code.INVALID_REQUEST || status.getCode() == NotificationStatus.Code.ERROR) {
            errorLogStatus(status, request, response);
        } else {
            debugLogStatus(status, request, response);
        }
    }

    private void debugLogStatus(NotificationStatus status, JnsSendMessageRequest request, HttpResponse<?> response) {
        logger.debug("{} - for {} request {} response {}",
                status,
                notification.getId(),
                request,
                response
        );
    }

    private void errorLogStatus(NotificationStatus status, JnsSendMessageRequest request, HttpResponse<?> response) {
        logger.warn("{} - for {} request {} response {}",
                status,
                notification.getId(),
                request,
                response);
    }

    private NotificationStatus classifyError(Throwable e, JnsSendMessageRequest request) {
        NotificationStatus status = NotificationStatus.ERROR;
        var root = Throwables.getRootCause(e);
        if (root instanceof HttpJnsClient.Error error) {
            HttpResponse<String> response = error.getResponse();
            String description = response.body().isBlank() ? Throwables.getStackTraceAsString(e) : response.body();
            if (HttpStatus.is2xx(response.statusCode())) {
                status = NotificationStatus.SUCCESS;
            }
            if (response.statusCode() == HttpStatus.SC_403_FORBIDDEN ||
                    response.statusCode() == HttpStatus.SC_401_UNAUTHORIZED) {
                status = NotificationStatus.PERMISSION_DENIED.withDescription(description);
            }
            if (response.statusCode() == HttpStatus.SC_429_TOO_MANY_REQUESTS) {
                status = NotificationStatus.RESOURCE_EXHAUSTED.withDescription(description);
            }
            if (HttpStatus.is4xx(response.statusCode())) {
                status = NotificationStatus.INVALID_REQUEST.withDescription(description);
            }
            if (HttpStatus.is5xx(response.statusCode())) {
                status = NotificationStatus.ERROR_ABLE_TO_RETRY.withDescription(description);
            }
            logStatus(status, request, error.getResponse());
        }
        return status.withDescription(Throwables.getStackTraceAsString(e));
    }
}
