package ru.yandex.solomon.alert.notification.channel.sms;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.yasms.YasmsClient;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
public class SmsClientImpl implements SmsClient {
    private final YasmsClient yasmsClient;
    private final UserPhoneProvider userPhoneProvider;

    public SmsClientImpl(YasmsClient yasmsClient, UserPhoneProvider userPhoneProvider) {
        this.yasmsClient = yasmsClient;
        this.userPhoneProvider = userPhoneProvider;
    }

    @Override
    public CompletableFuture<NotificationStatus> sendToPhone(String phone, String text, String identity) {
        return yasmsClient.sendToPhone(phone, text, identity)
                .thenApply(status -> {
                    switch (status.getCode()) {
                        case SUCCESS:
                            return NotificationStatus.SUCCESS.withDescription(status.getDetails());
                        case INTERNAL_ERROR:
                            return NotificationStatus.ERROR_ABLE_TO_RETRY
                                    .withDescription(status.getDetails());
                        case LIMIT_EXCEEDED:
                        case UID_LIMITEXCEEDED:
                            return NotificationStatus.RESOURCE_EXHAUSTED
                                    .withDescription(status.getDetails())
                                    .withRetryAfterMillis(TimeUnit.DAYS.toMillis(1));
                        case NO_UID:
                        case NO_PHONE:
                        case BAD_PHONE:
                        case NO_ROUTE:
                        case NO_CURRENT:
                        case NO_RIGHTS:
                        case INVALID_REQUEST:
                        case NO_TEXT:
                        case NO_SENDER:
                            return NotificationStatus.INVALID_REQUEST.withDescription(status.getDetails());
                        default:
                            return NotificationStatus.ERROR.withDescription(status.getDetails());
                    }
                });
    }

    @Override
    public CompletableFuture<NotificationStatus> sendToUser(String login, String text, String identity) {
        return userPhoneProvider.getPhone(login)
                .thenCompose(result -> {
                    if (!result.isOk()) {
                        return completedFuture(result.getStatus());
                    }

                    return sendToPhone(result.getPhone(), text, identity);
                });
    }
}
