package ru.yandex.solomon.alert.notification.channel.sms;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillCloseWhenClosed;

import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.channel.AbstractNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.EventBatchCollector;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.channel.SubAlertEventBatchCollector;
import ru.yandex.solomon.alert.notification.domain.sms.SmsNotification;
import ru.yandex.solomon.alert.util.WindowRateLimit;

import static ru.yandex.misc.concurrent.CompletableFutures.whenComplete;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class SmsNotificationChannel extends AbstractNotificationChannel<SmsNotification> {
    /**
     * https://wiki.yandex-team.ru/passport/yasms/sendsms/#dlinasoobshhenija
     */
    private static final int MAX_MESSAGE_SIZE = 402;

    private final SmsClient client;
    private final SmsTemplate template;
    @WillCloseWhenClosed
    private final EventBatchCollector batchCollector;
    private final WindowRateLimit rateLimit;
    private final Function<String, CompletableFuture<NotificationStatus>> sendFn;

    SmsNotificationChannel(
            SmsNotification notification,
            SmsClient client,
            SmsTemplate template,
            ScheduledExecutorService executorService,
            int dailyLimit)
    {
        super(notification);
        this.client = client;
        this.template = template;
        this.batchCollector = new SubAlertEventBatchCollector(executorService, this::sendMany);
        this.rateLimit = new WindowRateLimit(dailyLimit, TimeUnit.DAYS);
        this.sendFn = notification.getPhone().isEmpty()
                ? this::sendToLogin
                : this::sendToPhone;
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.statusFiltering(
            config.getNotifyAboutStatusesOrDefault(notification.getNotifyAboutStatus()),
            config.getRepeatNotificationDelayOrDefault(notification.getRepeatNotifyDelay()));
    }

    @Nonnull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        if (batchCollector.supportBatch(event)) {
            return batchCollector.add(event);
        }

        return send(template.getText(event));
    }

    private void sendMany(List<Event> events, CompletableFuture<NotificationStatus> future) {
        whenComplete(send(template.getText(events)), future);
    }

    private CompletableFuture<NotificationStatus> send(String text) {
        long delayNanos = rateLimit.delayNanosToPermit();
        if (delayNanos > 0) {
            return CompletableFuture.completedFuture(NotificationStatus.RESOURCE_EXHAUSTED
                    .withDescription("Daily sms limit on one phone reached")
                    .withRetryAfterMillis(TimeUnit.NANOSECONDS.toMillis(delayNanos)));
        }

        if (text.length() > MAX_MESSAGE_SIZE) {
            text = text.substring(0, MAX_MESSAGE_SIZE - 3) + "...";
        }

        return sendFn.apply(text);
    }

    private CompletableFuture<NotificationStatus> sendToPhone(String text) {
        return client.sendToPhone(notification.getPhone(), text, notification.getId());
    }

    private CompletableFuture<NotificationStatus> sendToLogin(String text) {
        return client.sendToUser(notification.getLogin(), text, notification.getId());
    }

    @Override
    public void close() {
        batchCollector.close();
    }
}
