package ru.yandex.solomon.alert.notification.channel.telegram;

import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javax.annotation.concurrent.GuardedBy;

import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.solomon.alert.dao.TelegramDao;
import ru.yandex.solomon.alert.dao.TelegramRecord;


/**
 * @author alexlovkov
 **/
public class ChatIdResolverImpl implements ChatIdStorage {

    private static final Logger logger = LoggerFactory.getLogger(ChatIdResolverImpl.class);

    private final TelegramDao dao;
    private final ReadWriteLock rwLock;
    // telegram login -> chat
    @GuardedBy("rwLock")
    private BiMap<String, TelegramChat> telegramLoginToChatId;

    // group title -> chat
    @GuardedBy("rwLock")
    private BiMap<String, TelegramChat> groupTitleToChatId;

    public ChatIdResolverImpl(TelegramDao dao) {
        this.dao = dao;
        this.rwLock = new ReentrantReadWriteLock();
    }

    @Override
    public CompletableFuture<Void> loadAll() {
        return dao.findAll()
            .thenAccept(telegramRecords -> {
                try {
                    rwLock.writeLock().lock();
                    recordsToMap(telegramRecords);
                } finally {
                    rwLock.writeLock().unlock();
                }
            });
    }

    private void recordsToMap(List<TelegramRecord> records) {
        BiMap<String, TelegramChat> telegramLoginToChatId = HashBiMap.create(records.size());
        BiMap<String, TelegramChat> groupTitleToChatId = HashBiMap.create(records.size());
        for (var record : records) {
            TelegramChat chat = new TelegramChat(record.getChatId());
            if (record.isGroup()) {
                if (groupTitleToChatId.putIfAbsent(record.getName(), chat) != null) {
                    logger.error("we have two rows with the same name:{}", record.getName());
                }
            } else {
                String telegramLoginLowerCase = record.getName().toLowerCase(Locale.ROOT);
                if (telegramLoginToChatId.putIfAbsent(telegramLoginLowerCase, chat) != null) {
                    logger.error("we have two rows with the same name:{}", record.getName());
                }
            }
        }
        this.telegramLoginToChatId = telegramLoginToChatId;
        this.groupTitleToChatId = groupTitleToChatId;
    }

    @Override
    public long getChatIdByTelegramLogin(String telegramLogin) {
        try {
            rwLock.readLock().lock();
            checkState();
            TelegramChat result = telegramLoginToChatId.get(telegramLogin.toLowerCase(Locale.ROOT));
            return result == null ? 0 : result.getChatId();
        } finally {
            rwLock.readLock().unlock();
        }
    }

    @Override
    public long getChatIdByGroupTitle(String groupTitle) {
        try {
            rwLock.readLock().lock();
            checkState();
            TelegramChat result = groupTitleToChatId.get(groupTitle);
            return result == null ? 0 : result.getChatId();
        } finally {
            rwLock.readLock().unlock();
        }
    }

    @Override
    public void removeTelegramLogin(String telegramLogin) {
        try {
            rwLock.writeLock().lock();
            checkState();
            telegramLoginToChatId.remove(telegramLogin.toLowerCase(Locale.ROOT));
        } finally {
            rwLock.writeLock().unlock();
        }
    }

    @Override
    public void removeGroupTitle(String title) {
        try {
            rwLock.writeLock().lock();
            checkState();
            groupTitleToChatId.remove(title);
        } finally {
            rwLock.writeLock().unlock();
        }
    }


    @Override
    public void addTelegramLogin(long chatId, String telegramLogin) {
        try {
            rwLock.writeLock().lock();
            checkState();
            telegramLoginToChatId.forcePut(telegramLogin.toLowerCase(Locale.ROOT), new TelegramChat(chatId));
        } finally {
            rwLock.writeLock().unlock();
        }
    }

    @Override
    public void addGroupTitle(long chatId, String groupTitle) {
        try {
            rwLock.writeLock().lock();
            checkState();
            groupTitleToChatId.forcePut(groupTitle, new TelegramChat(chatId));
        } finally {
            rwLock.writeLock().unlock();
        }
    }

    @Override
    public List<String> listGroups() {
        try {
            rwLock.readLock().lock();
            checkState();
            return new ArrayList<>(groupTitleToChatId.keySet());
        } finally {
            rwLock.readLock().unlock();
        }
    }

    // XXX chatId can change!
    @Override
    public String resolveGroupTitle(long chatId) {
        try {
            rwLock.readLock().lock();
            checkState();
            return groupTitleToChatId.inverse().get(new TelegramChat(chatId));
        } finally {
            rwLock.readLock().unlock();
        }
    }

    private void checkState() {
        if (telegramLoginToChatId == null || groupTitleToChatId == null) {
            throw new IllegalStateException("resolver is not initialized yet");
        }
    }
}
