package ru.yandex.solomon.alert.notification.channel.telegram;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.EvaluationStatus;
import ru.yandex.solomon.alert.dao.TelegramEventRecord;
import ru.yandex.solomon.alert.notification.channel.telegramLike.Payload;
import ru.yandex.solomon.alert.telegram.dto.InlineKeyboard;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class KeyboardFactory {
    public KeyboardFactory() {
    }

    @Nullable
    InlineKeyboard makeKeyboardForEventRecord(TelegramEventRecord eventRecord) {
        String uuid = eventRecord.getId();
        var appearance = eventRecord.getEventAppearance();
        var status = eventRecord.getAlertStatusCode();
        var projectId = eventRecord.getProjectId();

        return makeKeyboard(projectId, appearance, status, uuid);
    }

    @Nullable
    InlineKeyboard makeKeyboardForPayload(Payload payload, EventAppearance appearance, String uuid) {
        var status = payload.event().getState().getStatus().getCode();
        var projectId = payload.event().getAlert().getProjectId();

        return makeKeyboard(projectId, appearance, status, uuid);
    }

    @Nullable
    private InlineKeyboard makeKeyboard(String projectId, EventAppearance appearance, EvaluationStatus.Code status, String uuid) {
        return switch (appearance) {
            case TEXT_ONLY -> makeKeyboardForTextOnly(projectId, status, uuid);
            case PHOTO_FAILED -> makeKeyboardForPhotoFailed(projectId, status, uuid);
            case PHOTO_EMPTY -> makeKeyboardForEmptyPhoto(projectId, status, uuid);
            case WITH_PHOTO -> makeKeyboardForFullPhoto(projectId, status, uuid);
            case MUTE_SET -> makeKeyboardForMuteSet(uuid);
            default -> null;
        };
    }

    private InlineKeyboard makeKeyboardForTextOnly(String projectId, EvaluationStatus.Code status, String uuid) {
        InlineKeyboard keyboard = new InlineKeyboard(1);
        var row = keyboard.getRow(0);

        row.add(ButtonType.MUTE.with(uuid));

        // This was originally forgotten for text only messages
        // Bring them back when mutes are enabled for project
        if (needsFalsePositiveButton(status)) {
            row.add(ButtonType.FALSE_POSITIVE.with(uuid));
        }

        return keyboard;
    }

    private InlineKeyboard makeKeyboardForPhotoFailed(String projectId, EvaluationStatus.Code status, String uuid) {
        // User did not receive the screenshot of the original event, so render PAST button also
        InlineKeyboard keyboard = new InlineKeyboard(2);
        var row = keyboard.getRow(0);

        row.add(ButtonType.NOW.with(uuid));
        row.add(ButtonType.PAST.with(uuid));
        if (needsFalsePositiveButton(status)) {
            row.add(ButtonType.FALSE_POSITIVE.with(uuid));
        }

        keyboard.getRow(1).add(ButtonType.MUTE.with(uuid));

        return keyboard;
    }

    private InlineKeyboard makeKeyboardForEmptyPhoto(String projectId, EvaluationStatus.Code status, String uuid) {
        // The event has empty screenshot, do not render NOW and PAST buttons
        boolean needsFalsePositive = needsFalsePositiveButton(status);

        InlineKeyboard keyboard = new InlineKeyboard(1);
        var row = keyboard.getRow(0);

        if (needsFalsePositive) {
            row.add(ButtonType.FALSE_POSITIVE.with(uuid));
        }
        row.add(ButtonType.MUTE.with(uuid));
        return keyboard;
    }

    private InlineKeyboard makeKeyboardForFullPhoto(String projectId, EvaluationStatus.Code status, String uuid) {
        InlineKeyboard keyboard = new InlineKeyboard(2);
        var row = keyboard.getRow(0);

        row.add(ButtonType.NOW.with(uuid));
        if (needsFalsePositiveButton(status)) {
            row.add(ButtonType.FALSE_POSITIVE.with(uuid));
        }

        keyboard.getRow(1).add(ButtonType.MUTE.with(uuid));

        return keyboard;
    }

    public InlineKeyboard makeKeyboardForDurationSelection(String uuid) {
        InlineKeyboard keyboard = new InlineKeyboard(2);
        keyboard.getRow(0).add(ButtonType.CANCEL.with(uuid));
        var row = keyboard.getRow(1);
        row.add(ButtonType.FOR_HOUR.with(uuid));
        row.add(ButtonType.TILL_MORNING.with(uuid));
        row.add(ButtonType.TILL_MONDAY.with(uuid));
        return keyboard;
    }

    public InlineKeyboard makeKeyboardForLabelsSelectors(String uuid) {
        InlineKeyboard keyboard = new InlineKeyboard(2);
        keyboard.getRow(0).add(ButtonType.CANCEL.with(uuid));
        var row = keyboard.getRow(1);
        row.add(ButtonType.ALL_SUBALERTS.with(uuid));
        row.add(ButtonType.REPLY_SELECTORS.with(uuid));
        return keyboard;
    }

    public InlineKeyboard makeKeyboardForMuteSet(String uuid) {
        InlineKeyboard keyboard = new InlineKeyboard(1);
        keyboard.getRow(0).add(ButtonType.UNMUTE.with(uuid));
        return keyboard;
    }

    private boolean needsFalsePositiveButton(EvaluationStatus.Code code) {
        return code == EvaluationStatus.Code.ALARM || code == EvaluationStatus.Code.WARN;
    }
}
