package ru.yandex.solomon.alert.notification.channel.telegram;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.solomon.alert.dao.TelegramEventsDao;


/**
 * @author alexlovkov
 **/
public class TelegramEventsCleaner implements AutoCloseable {
    private static final Logger logger = LoggerFactory.getLogger(TelegramEventsCleaner.class);

    private final TelegramEventsDao telegramEventsDao;
    private final long ttlMillis;
    private final ScheduledExecutorService timer;
    private final ActorWithFutureRunner actor;
    private ScheduledFuture<?> scheduledFuture;
    private volatile boolean closed = false;

    public TelegramEventsCleaner(TelegramEventsDao telegramEventsDao, Duration ttl, ScheduledExecutorService executor) {
        this.telegramEventsDao = telegramEventsDao;
        this.ttlMillis = ttl.toMillis();
        this.timer = executor;
        this.actor = new ActorWithFutureRunner(this::act, executor);
        scheduleNext(TimeUnit.MINUTES.toMillis(10L), TimeUnit.HOURS.toMillis(1L));
    }

    private CompletableFuture<?> act() {
        if (closed) {
            return CompletableFuture.completedFuture(null);
        }
        return cleanup()
            .whenComplete((ignore, e) -> {
                if (e != null) {
                    logger.error("Clean up older telegram events failed", e);
                    scheduleNext(Math.round(ttlMillis * 0.1), Math.round(ttlMillis * 0.3));
                } else {
                    scheduleNext(Math.round(ttlMillis * 0.8), ttlMillis);
                }
            });
    }

    private CompletableFuture<Void> cleanup() {
        try {
            return telegramEventsDao.deleteOlderThan(Instant.now().minusMillis(ttlMillis));
        } catch (Throwable e) {
            return CompletableFuture.failedFuture(e);
        }
    }

    private void scheduleNext(long minDelay, long maxDelay) {
        if (closed) {
            return;
        }

        final long rndDelayMillis = ThreadLocalRandom.current().nextLong(minDelay, maxDelay);
        if (logger.isDebugEnabled()) {
            logger.debug("schedule next cleanup after {} ms", rndDelayMillis);
        }

        scheduledFuture = timer.schedule(actor::schedule, rndDelayMillis, TimeUnit.MILLISECONDS);
    }

    @Override
    public void close() {
        closed = true;
        var copy = scheduledFuture;
        if (copy != null) {
            copy.cancel(false);
        }
    }
}
