package ru.yandex.solomon.alert.notification.channel.telegram;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.charts.ChartsClient;
import ru.yandex.solomon.alert.dao.TelegramEventsDao;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.channel.AbstractNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.channel.SubAlertEventBatchCollector;
import ru.yandex.solomon.alert.notification.channel.telegramLike.Payload;
import ru.yandex.solomon.alert.notification.channel.telegramLike.TelegramLikeTransport;
import ru.yandex.solomon.alert.notification.domain.telegram.TelegramNotification;
import ru.yandex.solomon.alert.telegram.TelegramClient;
import ru.yandex.solomon.alert.telegram.dto.TelegramSendMessage;
import ru.yandex.staff.StaffClient;
import ru.yandex.staff.exceptions.StaffNotFoundException;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static ru.yandex.misc.concurrent.CompletableFutures.whenComplete;


/**
 * @author alexlovkov
 **/
@ParametersAreNonnullByDefault
public class TelegramNotificationChannel extends AbstractNotificationChannel<TelegramNotification> {

    private static final Logger logger = LoggerFactory.getLogger(TelegramNotificationChannel.class);

    private final StaffClient staffClient;
    private final ChatIdStorage chatIdResolver;
    private final TelegramLimits telegramLimits;
    private final TelegramEventsDao telegramEventsDao;
    private final TelegramTemplate template;
    private final SubAlertEventBatchCollector batchCollector;
    private final TelegramLikeTransport transport;

    TelegramNotificationChannel(
        TelegramNotification notification,
        TelegramClient client,
        StaffClient staffClient,
        ChartsClient chartsClient,
        ChatIdStorage chatIdResolver,
        ScheduledExecutorService executorService,
        TelegramEventsDao telegramEventsDao,
        TelegramLimits telegramLimits,
        TelegramTemplate template)
    {
        super(notification);
        this.staffClient = staffClient;
        this.chatIdResolver = chatIdResolver;
        this.telegramEventsDao = telegramEventsDao;
        this.template = template;
        this.batchCollector = new SubAlertEventBatchCollector(executorService, this::sendMany);
        this.telegramLimits = telegramLimits;
        this.transport = new TelegramTransport(client, telegramLimits, chartsClient);
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.statusFiltering(
            config.getNotifyAboutStatusesOrDefault(notification.getNotifyAboutStatus()),
            config.getRepeatNotificationDelayOrDefault(notification.getRepeatNotifyDelay()));
    }

    @Nonnull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        if (batchCollector.supportBatch(event)) {
            return batchCollector.add(event);
        }
        var payload = EventFeatures.maybePhotoEvent(notification.isSendScreenshot(), event);
        return sendAndRegisterEvent(payload.withText(template.getText(event)));
    }

    private void sendMany(/* @NonEmpty */ List<Event> events, CompletableFuture<NotificationStatus> future) {
        var payload = EventFeatures.chooseMaybeOneEventForPhoto(notification.isSendScreenshot(), events);
        whenComplete(sendAndRegisterEvent(payload.withText(template.getText(events))), future);
    }

    @Nonnull
    private CompletableFuture<NotificationStatus> sendAndRegisterEvent(Payload payload) {
        var permission = telegramLimits.attempt();
        if (!permission.havePermit()) {
            return completedFuture(NotificationStatus.RESOURCE_EXHAUSTED
                .withDescription("too many requests per second for " + permission.limit())
                .withRetryAfterMillis(permission.waitMillis()));
        }
        if (StringUtils.isNotEmpty(notification.getLogin())) {
            return sendToStaffLogin(notification.getLogin(), payload);
        } else {
            return sendToGroup(notification.getChatId(), payload);
        }
    }

    private CompletableFuture<NotificationStatus> sendToGroup(long groupChatId, Payload payload) {
        // group could be deleted, or bot was kicked and added with errors (i.e this title exists already)
        if (chatIdResolver.resolveGroupTitle(groupChatId) == null) {
            logger.info("group with chatId:{} wasn't found in resolver, projectId={}, notificationId={}",
                    groupChatId, notification.getProjectId(), notification.getId());
            return completedFuture(NotificationStatus.INVALID_REQUEST
                    .withDescription("can't resolve group, bot should be in the group-chat"));
        }
        return sendAndRegisterEvent(groupChatId, payload);
    }

    private CompletableFuture<NotificationStatus> sendToStaffLogin(String staffLogin, Payload payload) {
        return staffClient.getUserInfo(staffLogin)
                .handle((user, ex) -> {
                    if (ex != null) {
                        Throwable t = CompletableFutures.unwrapCompletionException(ex);
                        if (t instanceof StaffNotFoundException) {
                            logger.error("Staff user {} not found: {}", staffLogin, t.getMessage());
                            return completedFuture(NotificationStatus.INVALID_REQUEST
                                    .withDescription("User " + staffLogin + " not found on staff"));
                        } else {
                            logger.error("Staff reported error while looking for user {}", staffLogin, t);
                            return completedFuture(NotificationStatus.ERROR
                                    .withDescription("Staff replied: " + t.getMessage()));
                        }
                    }
                    if (user.isDismissed()) {
                        logger.info("can't send notification to user:{} because he is dismissed",
                                staffLogin);
                        return completedFuture(NotificationStatus.INVALID_REQUEST
                                .withDescription("user:" + staffLogin + " is dismissed"));
                    }
                    List<String> telegramLogins = user.getTelegramLogins();
                    if (telegramLogins.isEmpty()) {
                        return completedFuture(NotificationStatus.INVALID_REQUEST
                                .withDescription("user:" + staffLogin + " doesn't have telegram login on staff"));
                    }
                    return sendToTelegramLogin(telegramLogins.get(0), payload);
                })
                .thenCompose(x -> x);
    }

    private CompletableFuture<NotificationStatus> sendToTelegramLogin(String telegramLogin, Payload payload) {
        long chatId = chatIdResolver.getChatIdByTelegramLogin(telegramLogin);
        if (chatId == 0) {
            logger.info("can't send notification to telegramLogin:{} because resolver doesn't have it", telegramLogin);
            return completedFuture(NotificationStatus.NOT_SUBSCRIBED
                .withDescription("telegramLogin:" + telegramLogin + " (staffLogin: " + notification.getLogin() +
                    ") is not known. Please say /start to the bot."));
        } else {
            return sendAndRegisterEvent(chatId, payload);
        }
    }

    private CompletableFuture<NotificationStatus> sendAndRegisterEvent(long chatId, Payload payload) {
        return transport.sendPayload("", "", chatId, payload)
                .thenCompose(psr -> {
                    TelegramSendMessage r = psr.result();
                    if (!r.isSuccess()) {
                        return completedFuture(TelegramLikeTransport.classifyStatus(r));
                    }
                    return telegramEventsDao.insert(psr.eventRecord())
                            .handle((aVoid, t) -> {
                                if (t != null) {
                                    logger.error("couldn't insert inside telegramEvents", t);
                                }
                                return TelegramLikeTransport.classifyStatus(r);
                            });
                });
    }

    @Override
    public void close() {
    }
}
