package ru.yandex.solomon.alert.notification.channel.telegram;

import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.solomon.alert.charts.ChartsClient;
import ru.yandex.solomon.alert.dao.TelegramEventsDao;
import ru.yandex.solomon.alert.notification.channel.NotificationChannel;
import ru.yandex.solomon.alert.notification.channel.NotificationChannelFactory;
import ru.yandex.solomon.alert.notification.domain.Notification;
import ru.yandex.solomon.alert.notification.domain.NotificationType;
import ru.yandex.solomon.alert.notification.domain.telegram.TelegramNotification;
import ru.yandex.solomon.alert.telegram.TelegramClient;
import ru.yandex.solomon.alert.template.TemplateFactory;
import ru.yandex.solomon.alert.util.RateLimit;
import ru.yandex.solomon.alert.util.RateLimiters;
import ru.yandex.staff.StaffClient;


/**
 * @author alexlovkov
 **/
@ParametersAreNonnullByDefault
public class TelegramNotificationChannelFactory implements NotificationChannelFactory {

    private final TelegramClient telegramClient;
    private final StaffClient staffClient;
    private final ChartsClient chartsClient;
    private final ChatIdStorage chatIdResolver;
    private final TemplateFactory templateFactory;
    private final TelegramTemplate template;
    private final RawTelegramLimits rawLimits;
    private final ScheduledExecutorService scheduledExecutor;
    private final RateLimit generalRateLimit;
    private final TelegramEventsDao telegramEventsDao;

    public TelegramNotificationChannelFactory(
        TelegramClient telegramClient,
        StaffClient staffClient,
        ChartsClient chartsClient,
        ChatIdStorage chatIdResolver,
        TemplateFactory templateFactory,
        TelegramTemplate template,
        RawTelegramLimits rawLimits,
        ScheduledExecutorService scheduledExecutor,
        TelegramEventsDao telegramEventsDao)
    {
        this.telegramClient = telegramClient;
        this.staffClient = staffClient;
        this.chartsClient = chartsClient;
        this.chatIdResolver = chatIdResolver;
        this.templateFactory = templateFactory;
        this.template = template;
        this.rawLimits = rawLimits;
        this.scheduledExecutor = scheduledExecutor;
        this.generalRateLimit = RateLimiters.create(rawLimits.getGeneralRateLimit());
        this.telegramEventsDao = telegramEventsDao;
    }

    @Override
    public Map<NotificationType, Integer> getSupportTypes() {
        // TELEGRAM is also handled by FallbackNotificationChannelFactory, so lower priority here
        return Map.of(NotificationType.TELEGRAM, 10);
    }

    @Nonnull
    @Override
    public NotificationChannel createChannel(Notification notification) {
        TelegramNotification telegramNotification = (TelegramNotification) notification;
        TelegramTemplate template = this.template.override(templateFactory, telegramNotification);
        boolean isGroup = StringUtils.isEmpty(telegramNotification.getLogin());
        TelegramLimits limits = TelegramLimits.create(generalRateLimit, rawLimits, isGroup);
        return new TelegramNotificationChannel(
            telegramNotification,
            telegramClient,
            staffClient,
            chartsClient,
            chatIdResolver,
            scheduledExecutor,
            telegramEventsDao,
            limits,
            template);
    }
}
