package ru.yandex.solomon.alert.notification.channel.telegram;

import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Try;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.charts.ChartsClient;
import ru.yandex.solomon.alert.charts.exceptions.ChartsEmptyResultException;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.telegramLike.Payload;
import ru.yandex.solomon.alert.notification.channel.telegramLike.PayloadSendResult;
import ru.yandex.solomon.alert.notification.channel.telegramLike.TelegramLikeTransport;
import ru.yandex.solomon.alert.telegram.TelegramClient;
import ru.yandex.solomon.alert.telegram.dto.ParseMode;
import ru.yandex.solomon.alert.telegram.dto.TelegramSendMessage;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class TelegramTransport implements TelegramLikeTransport {
    private static final Logger logger = LoggerFactory.getLogger(TelegramTransport.class);

    /**
     * telegram docs:
     * Photo caption (may also be used when resending photos by file_id), 0-1024 characters
     */
    private static final int MAX_CAPTION_SIZE_FOR_PHOTO = 1024;

    private final TelegramClient client;
    private final ChartsClient chartsClient;
    private final TelegramLimits telegramLimits;
    private final KeyboardFactory keyboardFactory;

    public TelegramTransport(
            TelegramClient client,
            TelegramLimits telegramLimits,
            ChartsClient chartsClient)
    {
        this.client = client;
        this.chartsClient = chartsClient;
        this.telegramLimits = telegramLimits;
        this.keyboardFactory = new KeyboardFactory();
    }

    @Override
    public CompletableFuture<PayloadSendResult> sendPayload(String staffLogin, String groupId, long chatId, Payload payload) {
        if (payload.isPhoto()) {
            return sendMessageWithPhoto(chatId, payload);
        } else {
            return sendTextOnlyMessage(chatId, payload);
        }
    }

    CompletableFuture<PayloadSendResult> sendMessageWithPhoto(long chatId, Payload payload) {
        if (payload.text().length() < MAX_CAPTION_SIZE_FOR_PHOTO) {
            return sendMessageWithPhotoAsSingle(chatId, payload);
        } else {
            return sendMessageWithPhotoAsPair(chatId, payload);
        }
    }

    CompletableFuture<PayloadSendResult> sendTextOnlyMessage(long chatId, Payload payload) {
        var uuid = makeUUID();
        var keyboard = keyboardFactory.makeKeyboardForPayload(payload, EventAppearance.TEXT_ONLY, uuid);
        return client.sendMessage(chatId, payload.text(), ParseMode.HTML, keyboard)
                .thenApply(r -> payload.sendResult(r, uuid, EventAppearance.TEXT_ONLY));
    }

    private CompletableFuture<PayloadSendResult> sendMessageWithPhotoAsPair(long chatId, Payload payload) {
        CompletableFuture<TelegramSendMessage> sendMessageFuture = client.sendMessage(chatId, payload.text(), ParseMode.HTML);

        var permission = telegramLimits.attempt();
        if (!permission.havePermit()) {
            logger.warn("could not send photo due to rate limiter: {}", permission);
            return sendMessageFuture.thenCompose(r -> {
                        var uuid = makeUUID();
                        if (r.isSuccess()) {
                            // try to append buttons to existing text message
                            var keyboard = keyboardFactory.makeKeyboardForPayload(payload, EventAppearance.PHOTO_FAILED, uuid);
                            return client.editMessageReplyMarkup(chatId, r.getMessageId(), keyboard)
                                    .handle((editMarkup, t) -> payload.sendResult(r, uuid, EventAppearance.PHOTO_FAILED));
                        }
                        return completedFuture(payload.sendResult(r, uuid, EventAppearance.PHOTO_FAILED));
                });
        }
        return sendMessageFuture.thenCompose(sendResult -> {
                    if (!sendResult.isSuccess()) {
                        return completedFuture(payload.sendResult(sendResult, "", EventAppearance.PHOTO_FAILED));
                    }
                    return sendPhotoWithFallback(chatId, sendResult.getMessageId(), payload, appearance -> {
                        var uuid = makeUUID();
                        // try to append buttons to existing text message
                        var keyboard = keyboardFactory.makeKeyboardForPayload(payload, appearance, uuid);
                        return client.editMessageReplyMarkup(chatId, sendResult.getMessageId(), keyboard)
                                .handle((editMarkup, t) -> payload.sendResult(sendResult, uuid, appearance));
                    });
                });
    }

    private CompletableFuture<PayloadSendResult> sendMessageWithPhotoAsSingle(long chatId, Payload payload) {
        return sendPhotoWithFallback(chatId, 0, payload, appearance -> {
            var uuid = makeUUID();
            // try to append buttons to existing text message
            var keyboard = keyboardFactory.makeKeyboardForPayload(payload, appearance, uuid);
            return client.sendMessage(chatId, payload.text(), ParseMode.HTML, keyboard)
                    .thenApply(r -> payload.sendResult(r, uuid, appearance));
        });
    }

    private CompletableFuture<PayloadSendResult> sendPhotoWithFallback(
            long chatId,
            long replyMessageId,
            Payload payload,
            Function<EventAppearance, CompletableFuture<PayloadSendResult>> sendText)
    {
        Event event = payload.event();
        String text = replyMessageId == 0 ? payload.text() : null;
        return getPhoto(event)
                .thenCompose(maybePhoto -> {
                    if (maybePhoto.isFailure()) {
                        Throwable throwable = CompletableFutures.unwrapCompletionException(maybePhoto.getThrowable());
                        if (throwable instanceof ChartsEmptyResultException) {
                            logger.debug("empty photo for alert:{}", event.getAlertApiKey(), throwable);
                            return sendText.apply(EventAppearance.PHOTO_EMPTY);
                        } else {
                            logger.warn("couldn't get photo for alert:{}", event.getAlertApiKey(), throwable);
                            return sendText.apply(EventAppearance.PHOTO_FAILED);
                        }
                    }
                    var photo = maybePhoto.get();
                    var uuid = makeUUID();
                    var keyboard = keyboardFactory.makeKeyboardForPayload(payload, EventAppearance.WITH_PHOTO, uuid);
                    return client.sendPhoto(chatId, photo, text, ParseMode.HTML, replyMessageId, keyboard)
                            .thenCompose(r -> {
                                if (!r.isSuccess() && replyMessageId != 0) {
                                    // sendText is just a completedFuture with result
                                    return sendText.apply(EventAppearance.PHOTO_FAILED);
                                }
                                return completedFuture(payload.sendResult(r, uuid, EventAppearance.WITH_PHOTO));
                            });
                });
    }

    private CompletableFuture<Try<byte[]>> getPhoto(Event event) {
        return chartsClient.getScreenshot(event.getAlertApiKey(), event.getState().getLatestEval())
                .handle(Try::successOrFailure);
    }

    private static String makeUUID() {
        return UUID.randomUUID().toString();
    }
}
