package ru.yandex.solomon.alert.notification.channel.telegram;

import java.nio.ByteBuffer;
import java.time.DayOfWeek;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.format.SignStyle;
import java.time.temporal.ChronoUnit;
import java.time.temporal.TemporalAdjusters;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAccumulator;

import javax.annotation.Nullable;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.primitives.Longs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Try;
import ru.yandex.misc.actor.ActorWithFutureRunner;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.alert.charts.ChartsClient;
import ru.yandex.solomon.alert.charts.exceptions.ChartsBadRequestException;
import ru.yandex.solomon.alert.charts.exceptions.ChartsEmptyResultException;
import ru.yandex.solomon.alert.client.MuteApi;
import ru.yandex.solomon.alert.dao.TelegramDao;
import ru.yandex.solomon.alert.dao.TelegramEventRecord;
import ru.yandex.solomon.alert.dao.TelegramEventsDao;
import ru.yandex.solomon.alert.dao.TelegramRecord;
import ru.yandex.solomon.alert.notification.TemplateVarsFactory;
import ru.yandex.solomon.alert.protobuf.CreateMuteRequest;
import ru.yandex.solomon.alert.protobuf.ReadMuteRequest;
import ru.yandex.solomon.alert.protobuf.UpdateMuteRequest;
import ru.yandex.solomon.alert.protobuf.mute.Mute;
import ru.yandex.solomon.alert.protobuf.mute.SelectorsType;
import ru.yandex.solomon.alert.telegram.TelegramClient;
import ru.yandex.solomon.alert.telegram.dto.CallbackQuery;
import ru.yandex.solomon.alert.telegram.dto.ChatMemberUpdated;
import ru.yandex.solomon.alert.telegram.dto.ParseMode;
import ru.yandex.solomon.alert.telegram.dto.TelegramChat;
import ru.yandex.solomon.alert.telegram.dto.TelegramMember;
import ru.yandex.solomon.alert.telegram.dto.TelegramMessage;
import ru.yandex.solomon.alert.telegram.dto.TelegramSendMessage;
import ru.yandex.solomon.alert.telegram.dto.TelegramUpdate;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.alert.TelegramChannelConfig;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.locks.DistributedLock;
import ru.yandex.solomon.locks.LockDetail;
import ru.yandex.solomon.model.protobuf.MatchType;
import ru.yandex.solomon.model.protobuf.Selector;
import ru.yandex.solomon.util.collection.Nullables;
import ru.yandex.solomon.util.time.DurationUtils;
import ru.yandex.staff.StaffClient;

import static java.time.temporal.ChronoField.DAY_OF_MONTH;
import static java.time.temporal.ChronoField.DAY_OF_WEEK;
import static java.time.temporal.ChronoField.HOUR_OF_DAY;
import static java.time.temporal.ChronoField.MINUTE_OF_HOUR;
import static java.time.temporal.ChronoField.MONTH_OF_YEAR;
import static java.time.temporal.ChronoField.SECOND_OF_MINUTE;
import static java.time.temporal.ChronoField.YEAR;
import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.toList;


/**
 * @author alexlovkov
 **/
public class TelegramUpdatesReceiver implements AutoCloseable {

    private static final Logger logger = LoggerFactory.getLogger(TelegramUpdatesReceiver.class);
    private static final ZoneId MSK_ZONE = ZoneId.of("Europe/Moscow");
    private static final int MORNING_HOURS = 10; // am

    private static final DateTimeFormatter MOSCOW_DATE_TIME;
    static {
        // manually code maps to ensure correct data always used
        // (locale data can be changed by application code)
        Map<Long, String> dow = new HashMap<>();
        dow.put(1L, "Mon");
        dow.put(2L, "Tue");
        dow.put(3L, "Wed");
        dow.put(4L, "Thu");
        dow.put(5L, "Fri");
        dow.put(6L, "Sat");
        dow.put(7L, "Sun");
        Map<Long, String> moy = new HashMap<>();
        moy.put(1L, "Jan");
        moy.put(2L, "Feb");
        moy.put(3L, "Mar");
        moy.put(4L, "Apr");
        moy.put(5L, "May");
        moy.put(6L, "Jun");
        moy.put(7L, "Jul");
        moy.put(8L, "Aug");
        moy.put(9L, "Sep");
        moy.put(10L, "Oct");
        moy.put(11L, "Nov");
        moy.put(12L, "Dec");
        MOSCOW_DATE_TIME = new DateTimeFormatterBuilder()
                .parseCaseInsensitive()
                .parseLenient()
                .optionalStart()
                .appendText(DAY_OF_WEEK, dow)
                .appendLiteral(", ")
                .optionalEnd()
                .appendValue(DAY_OF_MONTH, 1, 2, SignStyle.NOT_NEGATIVE)
                .appendLiteral(' ')
                .appendText(MONTH_OF_YEAR, moy)
                .appendLiteral(' ')
                .appendValue(YEAR, 4)  // 2 digit year not handled
                .appendLiteral(' ')
                .appendValue(HOUR_OF_DAY, 2)
                .appendLiteral(':')
                .appendValue(MINUTE_OF_HOUR, 2)
                .optionalStart()
                .appendLiteral(':')
                .appendValue(SECOND_OF_MINUTE, 2)
                .optionalEnd()
                .appendLiteral(" MSK")
                .toFormatter(Locale.ROOT);
    }

    private final TelegramClient client;
    private final TelegramChannelConfig config;
    private final TelegramDao dao;
    private final TemplateVarsFactory templateVarsFactory;
    private final UpdatesReceiverAuthSupport authSupport;
    private final DistributedLock distributedLock;
    @SuppressWarnings("FieldCanBeLocal")
    private final ActorWithFutureRunner actor;
    private final ChatIdStorage resolver;
    private final TelegramEventsDao telegramEventsDao;
    private final ChartsClient chartsClient;
    private final MetricRegistry registry;
    private final LongAccumulator lastSeenUpdateId;
    private final AtomicInteger retryNumber = new AtomicInteger(0);
    private final Histogram processUpdateElapsedMs;
    private final int longPollTimeoutSeconds;
    private final MuteApi muteApi;
    private final KeyboardFactory keyboardFactory;

    private volatile ScheduledFuture<?> scheduledFuture;
    private final ScheduledExecutorService scheduledExecutorService;
    private volatile boolean loaded;
    private volatile long lastSeenLeaderSeqNo;
    private volatile boolean closed = false;

    public TelegramUpdatesReceiver(
        TelegramClient client,
        ScheduledExecutorService scheduledExecutorService,
        ExecutorService executor,
        TelegramChannelConfig config,
        TelegramDao dao,
        StaffClient staffClient,
        DistributedLock distributedLock,
        ChatIdStorage resolver,
        TelegramEventsDao telegramEventsDao,
        ChartsClient chartsClient,
        TemplateVarsFactory templateVarsFactory,
        MuteApi muteApi)
    {
        this.client = client;
        this.config = config;
        this.dao = dao;
        this.authSupport = new UpdatesReceiverAuthSupport(staffClient, client);
        this.distributedLock = distributedLock;
        this.resolver = resolver;
        this.telegramEventsDao = telegramEventsDao;
        this.chartsClient = chartsClient;
        this.lastSeenUpdateId = new LongAccumulator(Long::max, 0);
        this.loaded = false;
        this.actor = new ActorWithFutureRunner(this::act, executor);
        this.longPollTimeoutSeconds = Math.toIntExact(TimeConverter.protoToDuration(config.getFetchPeriod()).toSeconds());
        this.scheduledExecutorService = scheduledExecutorService;
        this.scheduledFuture = scheduledExecutorService.schedule(actor::schedule, 0, TimeUnit.MILLISECONDS);
        this.registry = MetricRegistry.root();
        registry.lazyGaugeInt64("telegramUpdatesReceiver.lastUpdateId", lastSeenUpdateId::get);
        registry.lazyRate("telegramUpdatesReceiver.updates_processed", lastSeenUpdateId::get);
        this.processUpdateElapsedMs = registry.histogramRate("telegramUpdatesReceiver.process_update_elapsed_ms",
                Histograms.exponential(13, 2, 16));
        this.templateVarsFactory = templateVarsFactory;
        this.muteApi = muteApi;
        this.keyboardFactory = new KeyboardFactory();
    }

    private CompletableFuture<?> act() {
        if (closed) {
            return completedFuture(null);
        }

        return migrateSchema()
            .thenCompose(ignore -> reload())
            .thenCompose(ignore -> receiveUpdates())
            .handle((aVoid, throwable) -> scheduleNextAct(throwable));
    }

    private Void scheduleNextAct(Throwable throwable) {
        int retryNumber;
        if (throwable != null) {
            logger.error("Unhandled exception in updates receive loop", throwable);
            retryNumber = this.retryNumber.addAndGet(1);
        } else {
            this.retryNumber.set(0);
            retryNumber = 0;
        }
        if (distributedLock.isLockedByMe()) {
            return scheduleNextActForLeader(retryNumber);
        } else {
            return scheduleNextActForFollower();
        }
    }

    private Void scheduleNextActForLeader(int retryNumber) {
        // leader is constantly polling for updates and does not read dao, so schedule it often
        long minimalDelayMillis = 100;
        long delayMillis = DurationUtils.backoff(minimalDelayMillis, TimeUnit.SECONDS.toMillis(longPollTimeoutSeconds), retryNumber);
        this.scheduledFuture = scheduledExecutorService.schedule(actor::schedule, delayMillis, TimeUnit.MILLISECONDS);

        return null;
    }

    private Void scheduleNextActForFollower() {
        // followers are reading dao only so schedule them as rare as is acceptable
        long refreshMillis = TimeUnit.SECONDS.toMillis(longPollTimeoutSeconds);
        long delayMillis = DurationUtils.randomize(refreshMillis);
        this.scheduledFuture = scheduledExecutorService.schedule(actor::schedule, delayMillis, TimeUnit.MILLISECONDS);

        return null;
    }

    private CompletableFuture<Void> migrateSchema() {
        if (!distributedLock.isLockedByMe()) {
            return completedFuture(null);
        }

        long currentSeqNo = distributedLock.lockDetail().map(LockDetail::seqNo).orElse(0L);
        if (lastSeenLeaderSeqNo == currentSeqNo) {
            return completedFuture(null);
        }

        return dao.migrate()
            .thenAccept(ignore -> lastSeenLeaderSeqNo = currentSeqNo);
    }

    private CompletableFuture<Void> reload() {
        if (!distributedLock.isLockedByMe()) {
            return resolver.loadAll();
        }

        if (loaded) {
            return completedFuture(null);
        }

        return resolver.loadAll()
            .whenComplete((v, t) -> {
                if (t != null) {
                    logger.error("couldn't load telegram records from dao", t);
                } else {
                    loaded = true;
                }
            });
    }

    private CompletableFuture<?> receiveUpdates() {
        if (!distributedLock.isLockedByMe()) {
            return completedFuture(null);
        }

        return client.getUpdates(lastSeenUpdateId.get() + 1, 100, longPollTimeoutSeconds)
            .thenCompose((updates) -> updates.stream()
                .map(update -> CompletableFutures.safeCall(() -> process(update))
                        .exceptionally(e -> {
                            logger.error("couldn't process update", e);
                            return null;
                        }))
                .collect(collectingAndThen(toList(), CompletableFutures::allOfUnit)));
    }

    private CompletableFuture<Void> process(TelegramUpdate update) {
        long startMillis = System.currentTimeMillis();
        var future = completedFuture(update)
                .thenCompose(this::processImpl)
                .whenComplete((r, t) -> {
                    long elapsedMillis = System.currentTimeMillis() - startMillis;
                    processUpdateElapsedMs.record(elapsedMillis);
                });
        lastSeenUpdateId.accumulate(update.getId());
        return future;
    }

    private static <T, U> CompletableFuture<T> whenCompleteCompose(CompletableFuture<T> stage, Callable<CompletableFuture<U>> afterComplete) {
        return stage
                .handle(Try::successOrFailure)
                .thenCompose(prev -> CompletableFutures.safeCall(afterComplete)
                        .handle((ignore1, ignore2) -> prev))
                .thenCompose(Try::toCompletedFuture);

    }

    // TODO: filter only interesting updates, and if any exceptions happen - upsert these updates into DAO with TTL
    private CompletableFuture<Void> processImpl(TelegramUpdate update) {
        CallbackQuery callbackQuery = update.getCallback();
        if (callbackQuery != null) {
            return whenCompleteCompose(processCallbackQuery(callbackQuery), () -> {
                String callbackId = Nullables.orEmpty(callbackQuery.getCallbackId());
                logger.debug("answering callback_id {}", callbackId);
                return client.answerCallbackQuery(callbackId);
            });
        }

        TelegramMessage message = update.getMessage();
        if (message != null) {
            return processMessage(message);
        }

        ChatMemberUpdated chatMemberUpdated = update.getChatMemberUpdated();
        if (chatMemberUpdated != null) {
            return processChatMemberUpdated(chatMemberUpdated);
        }

        logger.debug("telegram update {} with empty message", update.getId());
        return completedFuture(null);
    }

    private CompletableFuture<Void> processMessage(TelegramMessage message) {
        String text = message.getText();
        if (text != null) {
            return processTextMessage(message, text);
        }

        String newChatTitle = message.getNewChatTitle();
        if (newChatTitle != null) {
            return processNewChatTitle(message, newChatTitle);
        }

        TelegramMember leftMember = message.getLeftMember();
        if (leftMember != null) {
            return processLeftMember(leftMember, message.getChat());
        }

        TelegramMember newChatMember = message.getNewChatMember();
        if (newChatMember != null) {
            return processNewChatMember(newChatMember, message);
        }

        long migratedChatId = message.getMigrateFromChatId();
        if (migratedChatId != 0) {
            return processMigrateFromChat(message, migratedChatId);
        }

        logger.debug("skipped telegram message: {}", message);
        return completedFuture(null);
    }

    private CompletableFuture<Void> processChatMemberUpdated(ChatMemberUpdated chatMemberUpdated) {
        var newChatMember = chatMemberUpdated.getNewChatMember();
        if (!isMe(newChatMember.getUser())) {
            return completedFuture(null);
        }
        if (newChatMember.getStatus().equals("left")) {
            return processLeftMember(newChatMember.getUser(), chatMemberUpdated.getChat());
        } else {
            return processAddMemberToChatWithValidation(chatMemberUpdated.getFrom(), chatMemberUpdated.getChat());
        }
    }

    private CompletableFuture<Void> processNewChatMember(TelegramMember newChatMember, TelegramMessage message) {
        if (!isMe(newChatMember)) {
            return completedFuture(null);
        }
        return processAddMemberToChatWithValidation(message.getTelegramFrom(), message.getChat());
    }

    private CompletableFuture<Void> processCallbackQuery(CallbackQuery callbackQuery) {
        ButtonType.DecodedCallbackData buttonWithUuid;
        try {
            buttonWithUuid = ButtonType.decode(callbackQuery.getCallbackData());
        } catch (Exception e) {
            logger.error("Exception while decoding callback data {}, malformed callback?", callbackQuery.getCallbackData(), e);
            return completedFuture(null);
        }

        ButtonType button = buttonWithUuid.buttonType();
        String uuid = buttonWithUuid.uuid();

        return telegramEventsDao.find(uuid)
            .thenCompose(maybeTelegramEventRecord -> {
                long chatId = callbackQuery.getTelegramMessage().getChat().getId();
                String userTelegramLogin = Nullables.orEmpty(Nullables.map(callbackQuery.getTelegramFrom(), TelegramMember::getUserName));

                registry.rate("telegramUpdatesReceiver.buttonType", Labels.of("type", button.name())).inc();

                if (maybeTelegramEventRecord.isEmpty()) {
                    logger.warn("don't have event with id:{} in dao", uuid);
                    return client.sendMessage(chatId, "notification event is missing (maybe it is just too old)", ParseMode.PLAIN,
                            callbackQuery.getTelegramMessage().getId())
                            .thenAccept(r -> {});
                }
                TelegramEventRecord event = maybeTelegramEventRecord.get();

                return switch (button) {
                    case NOW, PAST -> processNowPastButtons(chatId, event, button);
                    case FALSE_POSITIVE -> processFalsePositiveButton(event);
                    case MUTE, UNMUTE -> processMuteButtons(userTelegramLogin, chatId, event, button);
                    case FOR_HOUR, TILL_MORNING, TILL_MONDAY -> processDurationButtons(userTelegramLogin, chatId, event, button);
                    case ALL_SUBALERTS -> processAllSubAlertsButton(chatId, event);
                    case REPLY_SELECTORS -> processReplySelectorsButton(chatId, event);
                    case CANCEL -> processCancelButton(chatId, event);
                };
            });
    }

    private CompletableFuture<Void> processNowPastButtons(long chatId, TelegramEventRecord event, ButtonType button) {
        Instant time = (button == ButtonType.NOW) ?  Instant.now() : Instant.ofEpochMilli(event.getEvaluatedAt());
        CompletableFuture<byte[]> photo = chartsClient.getScreenshot(event.getAlertApiKey(), time);
        return photo.handle(Try::successOrFailure)
                .thenCompose(maybePhoto -> handleReplyWithPhoto(maybePhoto, chatId, event))
                .thenAccept(r -> {});
    }

    private CompletableFuture<TelegramSendMessage> handleReplyWithPhoto(Try<byte[]> maybePhoto, long chatId, TelegramEventRecord event) {
        if (maybePhoto.isSuccess()) {
            var ph = maybePhoto.get();
            logger.info("send photo to:{} msgId:{}", chatId, event.getMessageId());
            return client.sendPhoto(chatId, ph, null, null, event.getMessageId(), null);
        }

        Throwable throwable = CompletableFutures.unwrapCompletionException(maybePhoto.getThrowable());
        if (throwable instanceof ChartsBadRequestException badRequest) {
            logger.error("failed to answer with photo to:{} msgId:{} requestId:{} traceId:{}",
                    chatId, event.getMessageId(), badRequest.getRequestId(), badRequest.getTraceId(), badRequest);
            if (badRequest instanceof ChartsEmptyResultException) {
                return client.sendMessage(chatId, "Alert chart is empty", ParseMode.PLAIN, event.getMessageId());
            }
        } else {
            logger.error("failed to answer with photo to:{} msgId:{}", chatId, event.getMessageId(), throwable);
        }
        return client.sendMessage(chatId, "Failed to take screenshot", ParseMode.PLAIN, event.getMessageId());
    }

    private CompletableFuture<Void> processFalsePositiveButton(TelegramEventRecord event) {
        logger.info("event:{} was marked as false-positive", event);
        registry.rate("telegramUpdatesReceiver.falsePositiveAlerts", Labels.of("projectId", event.getProjectId())).inc();
        return CompletableFuture.completedFuture(null);
    }

    private CompletableFuture<Void> processMuteButtons(String userTelegramLogin, long chatId, TelegramEventRecord event, ButtonType button) {
        String projectId = event.getProjectId();
        return authSupport.authenticate(userTelegramLogin, chatId)
                .thenCompose(subject -> {
                    Permission permission = button == ButtonType.MUTE ? Permission.CONFIGS_CREATE : Permission.CONFIGS_UPDATE;
                    return authSupport.authorize(subject, permission, projectId)
                            .handle((aVoid, authorizationException) -> {
                                if (authorizationException != null) {
                                    String message = "Permission denied to " + subject.getUniqueId() +
                                            " to control mutes in project " + projectId + ": " + authorizationException.getMessage();
                                    return client.sendMessage(chatId, message, ParseMode.HTML)
                                            .thenAccept(r -> {});
                                }
                                return processMuteButtonsAuthorized(subject, chatId, event, button);
                            });
                })
                .handle((r, t) -> {
                    if (t != null) {
                        logger.warn("Exception while processing mute button {}", button, t);
                    }
                    return null;
                });
    }

    private CompletableFuture<Void> processMuteButtonsAuthorized(AuthSubject subject, long chatId, TelegramEventRecord event, ButtonType button) {
        if (button == ButtonType.MUTE) {
            return processMute(subject, chatId, event);
        } else if (button == ButtonType.UNMUTE) {
            return processUnmute(subject, chatId, event);
        } else {
            logger.error("Wrong event type passed to processMuteButtonsAuthorized: {}", button);
            return completedFuture(null);
        }
    }

    private CompletableFuture<Void> processMute(AuthSubject subject, long chatId, TelegramEventRecord event) {
        if (StringUtils.isEmpty(event.getSubAlertId())) {
            return processMuteForSimpleAlert(subject, chatId, event);
        } else {
            return processMuteForMultiAlert(subject, chatId, event);
        }
    }

    private CompletableFuture<Void> processMuteForSimpleAlert(AuthSubject subject, long chatId, TelegramEventRecord event) {
        var keyboard = keyboardFactory.makeKeyboardForDurationSelection(event.getId());
        return client.editMessageReplyMarkup(chatId, event.getMessageId(), keyboard)
                .thenAccept(r -> {});
    }

    private CompletableFuture<Void> processMuteForMultiAlert(AuthSubject subject, long chatId, TelegramEventRecord event) {
        var keyboard = keyboardFactory.makeKeyboardForLabelsSelectors(event.getId());
        return client.editMessageReplyMarkup(chatId, event.getMessageId(), keyboard)
                .thenAccept(r -> {});
    }

    private CompletableFuture<Instant> updateMuteToCurrentTime(AuthSubject subject, String projectId, String muteId) {
        Instant now = Instant.now();
        long nowMillis = now.toEpochMilli();
        return muteApi.readMute(ReadMuteRequest.newBuilder()
                        .setProjectId(projectId)
                        .setId(muteId)
                        .build(), 0)
                .thenApply(r -> r.getMute().toBuilder()
                        .setToMillis(nowMillis)
                        .setUpdatedAt(nowMillis)
                        .setUpdatedBy(subject.getUniqueId())
                        .build())
                .thenCompose(mute -> muteApi.updateMute(UpdateMuteRequest.newBuilder()
                                .setMute(mute)
                                .build(), 0))
                .thenApply(r -> now);
    }

    private CompletableFuture<Void> processUnmute(AuthSubject subject, long chatId, TelegramEventRecord event) {
        return updateMuteToCurrentTime(subject, event.getProjectId(), event.getMuteId())
                .handle((now, t) -> {
                    if (t != null) {
                        Throwable ex = CompletableFutures.unwrapCompletionException(t);
                        return client.sendMessage(chatId, "Failed to end mute: " + ex.getMessage(), ParseMode.PLAIN, event.getMessageId())
                                .thenAccept(ignore -> {});
                    }
                    return client.sendMessage(chatId, "Mute was ended by " +
                            subject.getUniqueId() + " on " + prettyTime(now), ParseMode.PLAIN, event.getMessageId())
                            .thenCompose(r -> {
                                TelegramEventRecord patched = TelegramEventRecord.forMuteResolved(event);
                                return telegramEventsDao.insert(patched)
                                        .thenCompose(aVoid -> client.editMessageReplyMarkup(chatId, event.getMessageId(), null))
                                        .thenAccept(ignore -> {});
                            });
                })
                .thenCompose(f -> f);
    }

    private CompletableFuture<Void> resetKeyboard(long chatId, TelegramEventRecord event) {
        return client.editMessageReplyMarkup(chatId, event.getMessageId(), keyboardFactory.makeKeyboardForEventRecord(event))
                .handle((r, t) -> null);
    }

    private CompletableFuture<Void> processCancelButton(long chatId, TelegramEventRecord event) {
        // reset buttons to original state
        return resetKeyboard(chatId, event);
    }

    private CompletableFuture<Void> processAllSubAlertsButton(long chatId, TelegramEventRecord event) {
        event.getContext().labelsSelectors = "";
        return telegramEventsDao.updateContext(event)
                .thenCompose(aVoid -> {
                    var keyboard = keyboardFactory.makeKeyboardForDurationSelection(event.getId());
                    return client.editMessageReplyMarkup(chatId, event.getMessageId(), keyboard)
                            .thenAccept(r -> {});
                });
    }

    private CompletableFuture<Void> processReplySelectorsButton(long chatId, TelegramEventRecord event) {
        return client.sendMessage(chatId, "Please reply to this message with subalert labels selectors " +
                        "(e.g. cluster=production, host='*vla*')", ParseMode.HTML, event.getMessageId())
                .thenCompose(r -> {
                    if (!r.isSuccess()) {
                        return completedFuture(null);
                    }
                    String uuid = uuidFromMessageId(chatId, r.getMessageId());
                    TelegramEventRecord record = TelegramEventRecord.forPrompt(uuid, Instant.now(), r.getMessageId(),
                            event.getId(), EventAppearance.PROMPT_SUBALERT_SELECTORS);
                    return telegramEventsDao.insert(record)
                            .thenCompose(aVoid -> resetKeyboard(chatId, event));
                });
    }

    private CompletableFuture<Void> processDurationButtons(String userTelegramLogin, long chatId, TelegramEventRecord event, ButtonType button) {
        String projectId = event.getProjectId();
        return authSupport.authenticate(userTelegramLogin, chatId)
                .thenCompose(subject -> authSupport.authorize(subject, Permission.CONFIGS_CREATE, projectId)
                        .handle((aVoid, authorizationException) -> {
                            if (authorizationException != null) {
                                String message = "Permission denied to " + subject.getUniqueId() +
                                        " to crete mutes in project " + projectId + ": " + authorizationException.getMessage();
                                return client.sendMessage(chatId, message, ParseMode.HTML)
                                        .thenAccept(r -> {});
                            }
                            return processDurationButtonsAuthorized(subject, chatId, event, button);
                        }))
                .handle((r, t) -> {
                    if (t != null) {
                        logger.warn("Exception while processing mute button {}", button, t);
                    }
                    return null;
                });
    }

    private static Instant nextMonday(Instant now) {
        ZoneOffset offset = MSK_ZONE.getRules().getOffset(now);
        LocalDateTime ld = LocalDateTime.ofInstant(now, MSK_ZONE).minus(MORNING_HOURS, ChronoUnit.HOURS);
        ld = ld.with(TemporalAdjusters.next(DayOfWeek.MONDAY)).with(LocalTime.MIN).plus(MORNING_HOURS, ChronoUnit.HOURS);
        return ld.toInstant(offset);
    }


    private static Instant nextMorning(Instant now) {
        ZoneOffset offset = MSK_ZONE.getRules().getOffset(now);
        LocalDateTime ld = LocalDateTime.ofInstant(now, MSK_ZONE).minus(MORNING_HOURS, ChronoUnit.HOURS);
        ld = ld.with(LocalTime.MIN).plus(MORNING_HOURS + 24, ChronoUnit.HOURS);
        return ld.toInstant(offset);
    }

    @VisibleForTesting
    static Instant next(Instant now, ButtonType button) {
        return switch (button) {
            case FOR_HOUR -> now.plus(1, ChronoUnit.HOURS);
            case TILL_MORNING -> nextMorning(now);
            case TILL_MONDAY -> nextMonday(now);
            default -> throw new IllegalArgumentException("Bad to specification for mute");
        };
    }

    private String prettyTime(Instant time) {
        return time.atZone(MSK_ZONE).truncatedTo(ChronoUnit.SECONDS)
                .format(MOSCOW_DATE_TIME);
    }

    private CompletableFuture<Void> processDurationButtonsAuthorized(AuthSubject subject, long chatId, TelegramEventRecord event, ButtonType button) {
        Instant now = Instant.now();
        var nowMillis = now.toEpochMilli();
        String staffLogin = subject.getUniqueId();
        var to = next(now, button);
        var toMillis = to.toEpochMilli();
        var labelSelectors = SelectorsFormat.parse(Nullables.orEmpty(event.getContext().labelsSelectors));
        String projectId = event.getProjectId();
        var request = CreateMuteRequest.newBuilder().setMute(Mute.newBuilder()
                        .setProjectId(projectId)
                        .setCreatedBy(staffLogin)
                        .setUpdatedBy(staffLogin)
                        .setCreatedAt(nowMillis)
                        .setUpdatedAt(nowMillis)
                        .setName("From telegram (" + event.getAlertId() + ")")
                        .setFromMillis(nowMillis)
                        .setToMillis(toMillis)
                        .setSelectors(SelectorsType.newBuilder()
                                .setAlertSelector(Selector.newBuilder()
                                        .setKey("alert")
                                        .setPattern(event.getAlertId())
                                        .setMatchType(MatchType.EXACT))
                                .setLabelSelectors(LabelSelectorConverter.selectorsToNewProto(labelSelectors)))
                ).build();
        return muteApi.createMute(request, 0)
                .handle((r, t) -> {
                    if (t != null) {
                        Throwable ex = CompletableFutures.unwrapCompletionException(t);
                        logger.error("Failed to create mute via telegram", ex);
                        String reply = "Failed to create mute: " + ex.getMessage();
                        return client.sendMessage(chatId, reply, ParseMode.HTML, event.getMessageId())
                                .thenAccept(ignore -> {});
                    }
                    var uuid = UUID.randomUUID().toString();
                    String muteId = r.getMute().getId();
                    String muteTill = prettyTime(to);
                    String msg = "Alert <a href=\"" + templateVarsFactory.makeMuteUrl(projectId, muteId) +
                            "\">muted</a> by " + staffLogin + " and will not send notifications till " + muteTill;
                    var keyboard = keyboardFactory.makeKeyboardForMuteSet(uuid);
                    return client.sendMessage(chatId, msg, ParseMode.HTML, keyboard, event.getMessageId())
                            .thenCompose(muteSetResponse -> {
                                if (!muteSetResponse.isSuccess()) {
                                    return completedFuture(null);
                                }
                                var record = TelegramEventRecord.forMuteCreated(uuid, now, muteSetResponse.getMessageId(),
                                        projectId, muteId);
                                return telegramEventsDao.insert(record);
                            });
                })
                .thenCompose(f -> f)
                .thenCompose(aVoid -> resetKeyboard(chatId, event)) // Hide mute menu
                .thenCompose(aVoid -> {
                    if (event.getEventAppearance() == EventAppearance.FORWARD) {
                        return client.editMessageText(chatId, event.getMessageId(),
                                "Selected duration: " + button.getCaption(), null);
                    }
                    return completedFuture(null);
                })
                .thenAccept(r -> {});
    }

    private CompletableFuture<Void> processLeftMember(TelegramMember leftMember, TelegramChat chat) {
        if (isMe(leftMember)) {
            resolver.removeGroupTitle(chat.getTitle());
            return dao.deleteById(chat.getId());
        }

        return completedFuture(null);
    }

    private CompletableFuture<Void> processMigrateFromChat(TelegramMessage message, long migratedChatId) {
        CompletableFuture<Void> futureDelete = completedFuture(null);
        String oldGroup = resolver.resolveGroupTitle(migratedChatId);
        if (oldGroup != null) {
            resolver.removeGroupTitle(oldGroup);
            futureDelete = dao.deleteById(migratedChatId);
        }
        return futureDelete.thenCompose(r -> processAddMemberToChat(message.getChat()));
    }

    private CompletableFuture<Void> processNewChatTitle(
        TelegramMessage message,
        String newChatTitle)
    {
        TelegramChat chat = message.getChat();
        long chatId = chat.getId();
        return dao.get(chatId).thenCompose(r -> {
            if (r.isEmpty()) {
                return processAddMemberToChatWithValidation(message.getTelegramFrom(), chat);
            } else {
                TelegramRecord oldRecord = r.get();
                if (resolver.getChatIdByGroupTitle(newChatTitle) == 0) {
                    resolver.removeGroupTitle(oldRecord.getName());
                    return dao.deleteById(oldRecord.getChatId())
                        .thenCompose((ignore) -> processAddMemberToChat(chat));
                } else {
                    return chatAlreadyExistMessage(chat);
                }
            }
        });
    }

    private CompletableFuture<Void> processAddMemberToChat(TelegramChat chat) {
        return insert(TelegramRecord.createForChat(chat.getId(), chat.getTitle()));
    }

    private CompletableFuture<Void> processAddMemberToChatWithValidation(
        @Nullable TelegramMember from,
        TelegramChat chat)
    {
        if (from == null) {
            return completedFuture(null);
        }
        String telegramLogin = Nullables.orEmpty(from.getUserName());
        return authSupport.authenticate(telegramLogin, chat.getId())
                .thenCompose(subject -> {
                    long oldChatId = resolver.getChatIdByGroupTitle(chat.getTitle());
                    if (oldChatId != 0) {
                        // something strange: bot was added to the chat twice without deletion
                        logger.warn("bot was added to the chat:{} oldChatId:{}", chat.getId(), oldChatId);
                        return chatAlreadyExistMessage(chat);
                    } else {
                        return processAddMemberToChat(chat);
                    }
                })
                .handle((r, t) -> {
                    if (t != null) {
                        logger.warn("Exception while processing adding member to chat", t);
                    }
                    return null;
                });
    }

    private CompletableFuture<Void> chatAlreadyExistMessage(TelegramChat chat) {
        return client.sendMessage(chat.getId(), "Unfortunately chat with name \"" + chat.getTitle() +
                        "\" already exist, you should rename your chat",
                        ParseMode.PLAIN)
                .thenAccept(r -> {});
    }

    private CompletableFuture<Void> processTextMessage(TelegramMessage msg, String text) {
        TelegramChat chat = msg.getChat();
        long chatId = chat.getId();

        TelegramMessage replyToMessage = msg.getReplyToMessage();
        if (replyToMessage == null) {
            return processTextMessageWithoutReply(chat, text);
        }

        long replyMessageId = replyToMessage.getId();
        String uuid = uuidFromMessageId(chatId, replyMessageId);
        return telegramEventsDao.find(uuid)
                .thenCompose(maybeEventRecord -> {
                    if (maybeEventRecord.isEmpty()) {
                        return processTextMessageWithoutReply(chat, text);
                    }
                    var record = maybeEventRecord.get();
                    String origUuid = record.getOrigEventUuid();
                    return processReplyToPrompt(record.getEventAppearance(), origUuid, chatId, msg, text);
                });
    }

    private CompletableFuture<Void> processReplyToPrompt(EventAppearance eventAppearance, String origUuid, long chatId, TelegramMessage msg, String text) {
        if (eventAppearance == EventAppearance.PROMPT_SUBALERT_SELECTORS) {
            return processReplyToPromptSubalertSelectors(origUuid, chatId, msg.getId(), text);
        }
        return completedFuture(null);
    }

    private static String validateSelectors(String text) {
        try {
            SelectorsFormat.parse(text);
        } catch (Exception e) {
            return e.getMessage();
        }
        return null;
    }

    private CompletableFuture<Void> processReplyToPromptSubalertSelectors(String origUuid, long chatId, long msgId, String text) {
        return telegramEventsDao.find(origUuid)
                .thenCompose(maybeEventRecord -> {
                    if (maybeEventRecord.isEmpty()) {
                        return completedFuture(null);
                    }
                    var event = maybeEventRecord.get();
                    String msg = validateSelectors(text);
                    if (msg != null) {
                        return client.sendMessage(chatId, "Selectors are invalid: " + msg, ParseMode.PLAIN, msgId)
                                .thenAccept(r -> {});
                    }
                    event.getContext().labelsSelectors = text;
                    var forwardUuid = UUID.randomUUID().toString();
                    var keyboard = keyboardFactory.makeKeyboardForDurationSelection(forwardUuid);

                    return client.sendMessage(chatId, "Select duration", ParseMode.PLAIN, keyboard, msgId)
                            .thenCompose(r -> {
                                if (!r.isSuccess()) {
                                    return completedFuture(null);
                                }
                                var forwardEvent = TelegramEventRecord.forward(forwardUuid,
                                        Instant.now(), r.getMessageId(), event);
                                return telegramEventsDao.insert(forwardEvent);
                            });
                });
    }

    private CompletableFuture<Void> processTextMessageWithoutReply(TelegramChat chat, String text) {
        String telegramLogin = Nullables.orEmpty(chat.getUserName());
        long chatId = chat.getId();
        if (chat.getType() != TelegramChat.Type.PRIVATE) {
            return completedFuture(null);
        }
        if (!text.equals("/start")) {
            return client.sendMessage(chatId, "Bot understands only /start command", ParseMode.PLAIN)
                    .thenAccept(r -> {});
        }
        return authSupport.authenticate(telegramLogin, chatId)
                .thenCompose(subject -> insert(TelegramRecord.createForUser(chatId, telegramLogin)))
                .handle((r, t) -> {
                    if (t != null) {
                        logger.warn("Exception while processing text message", t);
                    }
                    return null;
                });
    }

    // Reply flow does not contain any cookie-like value, so use UUID(chatId:msgId) as event id in dao
    private static String uuidFromMessageId(long chatId, long messageId) {
        var buffer = ByteBuffer.allocate(2 * Longs.BYTES);
        return UUID.nameUUIDFromBytes(buffer.putLong(chatId).putLong(messageId).array()).toString();
    }

    private boolean isMe(TelegramMember member) {
        return member.isBot()
            && member.getUserName().equals(config.getBotName());
    }

    private CompletableFuture<Void> insert(TelegramRecord record) {
        boolean isGroup = record.isGroup();
        String name = record.getName();
        long chatId = record.getChatId();
        if (isGroup) {
            resolver.addGroupTitle(chatId, name);
        } else {
            resolver.addTelegramLogin(chatId, name);
        }
        return dao.upsert(record)
            .thenCompose(ignore -> client.sendMessage(chatId, (isGroup ? "Group" : "User") +
                    " chat registered in Solomon", ParseMode.PLAIN))
            .thenAccept(r -> {});
    }

    @VisibleForTesting
    long getLastSeenUpdateId() {
        return lastSeenUpdateId.get();
    }

    @Override
    public void close() {
        closed = true;
        var scheduledFuture = this.scheduledFuture;
        if (scheduledFuture != null) {
            scheduledFuture.cancel(true);
        }
    }
}
