package ru.yandex.solomon.alert.notification.channel.telegram;

import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import javax.annotation.ParametersAreNonnullByDefault;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.alert.telegram.TelegramClient;
import ru.yandex.solomon.alert.telegram.dto.ParseMode;
import ru.yandex.solomon.auth.AuthSubject;
import ru.yandex.solomon.auth.exceptions.AuthenticationException;
import ru.yandex.solomon.auth.local.AsUserSubject;
import ru.yandex.solomon.auth.roles.Permission;
import ru.yandex.staff.StaffClient;
import ru.yandex.staff.UserInfo;
import ru.yandex.staff.exceptions.StaffClientException;
import ru.yandex.staff.exceptions.StaffNotFoundException;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class UpdatesReceiverAuthSupport {

    private static final Logger logger = LoggerFactory.getLogger(UpdatesReceiverAuthSupport.class);

    private final StaffClient staffClient;
    private final TelegramClient client;

    public UpdatesReceiverAuthSupport(StaffClient staffClient, TelegramClient client) {
        this.staffClient = staffClient;
        this.client = client;
    }

    CompletableFuture<AuthSubject> authenticate(String userTelegramLogin, long chatId) {
        if (userTelegramLogin.isEmpty()) {
            return failedFuture(new AuthenticationException("no telegram login"));
        }
        return staffClient.getUserInfoByTelegramLogin(userTelegramLogin)
                .handle((userInfo, t) -> {
                    if (t != null) {
                        return handleStaffException(t, chatId, userTelegramLogin)
                                .thenCompose(aVoid -> CompletableFuture.<AuthSubject>failedFuture(new AuthenticationException("staff exception")));
                    }
                    if (userInfo.isDismissed()) {
                        return handleDismissedUser(userTelegramLogin, userInfo)
                                .thenCompose(aVoid -> CompletableFuture.<AuthSubject>failedFuture(new AuthenticationException("user is dismissed")));
                    }
                    return CompletableFuture.<AuthSubject>completedFuture(new AsUserSubject(userInfo.getLogin()));
                })
                .thenCompose(f -> f);
    }

    // TODO: use real authorizer here
    // Should complete exceptionally if permission is denied
    CompletableFuture<Void> authorize(AuthSubject subject, Permission permission, String projectId) {
        return completedFuture(null);
    }

    private CompletableFuture<Void> handleDismissedUser(String telegramLogin, UserInfo userInfo) {
        logger.info("telegram login:{} is dismissed, staff login:{}", telegramLogin, userInfo.getLogin());
        return completedFuture(null);
    }

    private CompletableFuture<Void> handleStaffException(Throwable cft, long chatId, String telegramLogin) {
        Throwable t = CompletableFutures.unwrapCompletionException(cft);
        if (t instanceof StaffClientException e) {
            return handleStaffClientException(e, telegramLogin, chatId);
        }
        logger.error("Unhanded exception while interacting with Staff", t);
        if (t instanceof RuntimeException e) {
            throw e;
        }
        throw new RuntimeException(t);
    }

    private CompletableFuture<Void> handleStaffClientException(StaffClientException e, String telegramLogin, long chatId) {
        if (e instanceof StaffNotFoundException) {
            logger.info("staff wasn't found for:{}", telegramLogin);
            return client.sendMessage(chatId, "Telegram login " + telegramLogin +
                            " wasn't found at Yandex Staff, please add work-telegram to your staff account",
                    ParseMode.PLAIN
            ).thenAccept(x -> {});
        }
        UUID uuid = UUID.randomUUID();
        logger.error("Unhandled error from staff, resolved with {}", uuid, e);
        return client.sendMessage(chatId, "Error accessing Yandex Staff, try again later or contact Solomon support" +
                        " with request-id " + uuid,
                ParseMode.PLAIN
        ).thenAccept(x -> {});
    }
}
