package ru.yandex.solomon.alert.notification.channel.webhook;

import java.time.Instant;
import java.util.Map;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import com.google.common.base.MoreObjects;
import com.google.common.base.Throwables;
import com.google.common.net.MediaType;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.Request;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.misc.io.http.HttpHeaderNames;
import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.TemplateVarsFactory;
import ru.yandex.solomon.alert.notification.channel.AbstractNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.HttpHelper;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.domain.webhook.WebhookNotification;
import ru.yandex.solomon.alert.rule.EvaluationState;
import ru.yandex.solomon.alert.template.Template;

import static java.util.concurrent.CompletableFuture.completedFuture;

/**
 * @author Vladimir Gordiychuk
 */
@ParametersAreNonnullByDefault
public class WebhookNotificationChannel extends AbstractNotificationChannel<WebhookNotification> {
    private static final Logger logger = LoggerFactory.getLogger(WebhookNotificationChannel.class);
    private final AsyncHttpClient client;
    private final Template contentTemplate;
    private final TemplateVarsFactory variablesFactory;

    public WebhookNotificationChannel(
            WebhookNotification notification,
            @WillNotClose AsyncHttpClient client,
            TemplateVarsFactory variablesFactory,
            Template contentTemplate)
    {
        super(notification);
        this.client = client;
        this.variablesFactory = variablesFactory;
        this.contentTemplate = contentTemplate;
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.statusFiltering(
                config.getNotifyAboutStatusesOrDefault(notification.getNotifyAboutStatus()),
                config.getRepeatNotificationDelayOrDefault(notification.getRepeatNotifyDelay()));
    }

    @Nonnull
    @Override
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        EvaluationState state = event.getState();
        try {
            Request request = makeRequest(event);
            return performRequest(request);
        } catch (Throwable e) {
            errorLogStatus(NotificationStatus.INVALID_REQUEST, state, e);
            return completedFuture(NotificationStatus.INVALID_REQUEST.withDescription(Throwables.getStackTraceAsString(e)));
        }
    }

    @Override
    public void close() {
    }

    private CompletableFuture<NotificationStatus> performRequest(Request request) {
        return client.executeRequest(request)
                .toCompletableFuture()
                .thenApply(response -> handleResponse(request, response))
                .exceptionally(e -> {
                    errorLogStatus(NotificationStatus.ERROR, request, e);
                    return NotificationStatus.ERROR.withDescription(Throwables.getStackTraceAsString(e));
                });
    }

    private NotificationStatus handleResponse(Request request, Response response) {
        NotificationStatus status = HttpHelper.responseToStatus(response);
        logStatus(status, request, response);
        return status;
    }

    private Request makeRequest(Event event) {
        Map<String, Object> params = variablesFactory.makeTemplateParams(event);
        RequestBuilder request = new RequestBuilder("POST");
        request.setHeader(HttpHeaderNames.CONTENT_TYPE, MediaType.JSON_UTF_8.toString());
        notification.getHeaders().forEach(request::setHeader);
        return request.setUrl(notification.getUrl())
                .setBody(contentTemplate.process(params))
                .build();
    }

    private void debugLogStatus(NotificationStatus status, EvaluationState state) {
        logger.debug("{} - for {} by {}", status, notification.getId(), state);
    }

    private void errorLogStatus(NotificationStatus status, EvaluationState state, Throwable e) {
        logger.warn("{} - for {} by {}", status, notification.getId(), state, e);
    }

    private void logStatus(NotificationStatus status, Request request, Response response) {
        if (status.getCode() == NotificationStatus.Code.INVALID_REQUEST || status.getCode() == NotificationStatus.Code.ERROR) {
            errorLogStatus(status, request, response);
        } else {
            debugLogStatus(status, request, response);
        }
    }

    private void debugLogStatus(NotificationStatus status, Request request, Response response) {
        if (logger.isDebugEnabled()) {
            logger.debug("{} - for {} request {} response {}",
                    status,
                    notification.getId(),
                    toString(request),
                    toString(response)
            );
        }
    }

    private void errorLogStatus(NotificationStatus status, Request request, Response response) {
        logger.warn("{} - for {} request {} response {}",
                status,
                notification.getId(),
                toString(request),
                toString(response));
    }

    private void errorLogStatus(NotificationStatus status, Request request, Throwable e) {
        logger.warn("{} - for {} request {}", status, notification.getId(), toString(request), e);
    }

    private String toString(Request request) {
        return MoreObjects.toStringHelper(request)
                .omitNullValues()
                .add("method", request.getMethod())
                .add("url", request.getUrl())
                .add("headers", request.getHeaders().entries())
                .add("body", request.getStringData())
                .toString();
    }

    private String toString(Response response) {
        return MoreObjects.toStringHelper(response)
                .omitNullValues()
                .add("statusCode", response.getStatusCode())
                .add("statusText", response.getStatusText())
                .add("headers", response.getHeaders().entries())
                .add("body", response.getResponseBody())
                .toString();
    }
}
