package ru.yandex.solomon.alert.notification.channel.yachats;

import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.solomon.alert.domain.ChannelConfig;
import ru.yandex.solomon.alert.notification.DispatchRule;
import ru.yandex.solomon.alert.notification.DispatchRuleFactory;
import ru.yandex.solomon.alert.notification.channel.AbstractNotificationChannel;
import ru.yandex.solomon.alert.notification.channel.Event;
import ru.yandex.solomon.alert.notification.channel.NotificationStatus;
import ru.yandex.solomon.alert.notification.channel.SubAlertEventBatchCollector;
import ru.yandex.solomon.alert.notification.channel.telegramLike.Payload;
import ru.yandex.solomon.alert.notification.channel.telegramLike.TelegramLikeTransport;
import ru.yandex.solomon.alert.notification.domain.yachats.YaChatsNotification;
import ru.yandex.solomon.alert.yachats.YaChatsClient;

import static ru.yandex.misc.concurrent.CompletableFutures.whenComplete;

/**
 * @author Nikita Minin
 */
@ParametersAreNonnullByDefault
public class YaChatsNotificationChannel extends AbstractNotificationChannel<YaChatsNotification> {
    private final YaChatsLimits yaChatsLimits;
    private final YaChatsTemplate template;
    private final SubAlertEventBatchCollector batchCollector;
    private final TelegramLikeTransport transport;

    YaChatsNotificationChannel(
        YaChatsNotification notification,
        YaChatsClient client,
        ScheduledExecutorService executorService,
        YaChatsLimits yaChatsLimits,
        YaChatsTemplate template)
    {
        super(notification);
        this.template = template;
        this.batchCollector = new SubAlertEventBatchCollector(executorService, this::sendMany);
        this.yaChatsLimits = yaChatsLimits;
        this.transport = new YaChatsTransport(client);
    }

    @Override
    protected DispatchRule makeDispatchRule(ChannelConfig config) {
        return DispatchRuleFactory.statusFiltering(
            config.getNotifyAboutStatusesOrDefault(notification.getNotifyAboutStatus()),
            config.getRepeatNotificationDelayOrDefault(notification.getRepeatNotifyDelay()));
    }

    @Nonnull
    public CompletableFuture<NotificationStatus> send(Instant latestSuccessSend, Event event) {
        if (batchCollector.supportBatch(event)) {
            return batchCollector.add(event);
        }
        return send(Payload.ofTextEvent(event).withText(template.getText(event)));
    }

    @Nonnull
    private CompletableFuture<NotificationStatus> send(Payload payload) {
        YaChatsNotificationChannel.Permission permission = havePermit();
        if (!permission.havePermit) {
            return CompletableFuture.completedFuture(NotificationStatus.RESOURCE_EXHAUSTED
                .withDescription("too many requests per second for " + permission.limit)
                .withRetryAfterMillis(permission.waitMillis));
        }
        return sendMessage(payload);
    }

    private CompletableFuture<NotificationStatus> sendMessage(Payload payload) {
        return transport.sendPayload(notification.getLogin(), notification.getGroupId(), 0, payload)
                .thenApply(r -> TelegramLikeTransport.classifyStatus(r.result()));
    }

    private void sendMany(List<Event> events, CompletableFuture<NotificationStatus> future) {
        whenComplete(send(Payload.ofTextEvent(events.get(0)).withText(template.getText(events))), future);
    }

    @Override
    public void close() {
    }

    private YaChatsNotificationChannel.Permission havePermit() {
        if (!yaChatsLimits.getRateLimitPerChat().attempt()) {
            return new YaChatsNotificationChannel.Permission(false, TimeUnit.SECONDS.toMillis(1), "chat");
        }
        if (!yaChatsLimits.getGeneralRateLimit().attempt()) {
            return new YaChatsNotificationChannel.Permission(false, TimeUnit.SECONDS.toMillis(1), "common");
        }
        return new YaChatsNotificationChannel.Permission(true);
    }

    private static class Permission {

        private final String limit;
        private final boolean havePermit;
        private final long waitMillis;

        Permission(boolean havePermit) {
            this(havePermit, 0, "");
        }

        Permission(boolean havePermit, long waitMillis, String limit) {
            this.havePermit = havePermit;
            this.waitMillis = waitMillis;
            this.limit = limit;
        }
    }
}
